/*******************************************************************************
* File Name: ResetTimer.c
* Version 2.10
*
* Description:
*  This file provides the source code to the API for the ResetTimer
*  component
*
* Note:
*  None
*
********************************************************************************
* Copyright 2013-2015, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
*******************************************************************************/

#include "ResetTimer.h"

uint8 ResetTimer_initVar = 0u;


/*******************************************************************************
* Function Name: ResetTimer_Init
********************************************************************************
*
* Summary:
*  Initialize/Restore default ResetTimer configuration.
*
* Parameters:
*  None
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_Init(void)
{

    /* Set values from customizer to CTRL */
    #if (ResetTimer__QUAD == ResetTimer_CONFIG)
        ResetTimer_CONTROL_REG = ResetTimer_CTRL_QUAD_BASE_CONFIG;
        
        /* Set values from customizer to CTRL1 */
        ResetTimer_TRIG_CONTROL1_REG  = ResetTimer_QUAD_SIGNALS_MODES;

        /* Set values from customizer to INTR */
        ResetTimer_SetInterruptMode(ResetTimer_QUAD_INTERRUPT_MASK);
        
         /* Set other values */
        ResetTimer_SetCounterMode(ResetTimer_COUNT_DOWN);
        ResetTimer_WritePeriod(ResetTimer_QUAD_PERIOD_INIT_VALUE);
        ResetTimer_WriteCounter(ResetTimer_QUAD_PERIOD_INIT_VALUE);
    #endif  /* (ResetTimer__QUAD == ResetTimer_CONFIG) */

    #if (ResetTimer__TIMER == ResetTimer_CONFIG)
        ResetTimer_CONTROL_REG = ResetTimer_CTRL_TIMER_BASE_CONFIG;
        
        /* Set values from customizer to CTRL1 */
        ResetTimer_TRIG_CONTROL1_REG  = ResetTimer_TIMER_SIGNALS_MODES;
    
        /* Set values from customizer to INTR */
        ResetTimer_SetInterruptMode(ResetTimer_TC_INTERRUPT_MASK);
        
        /* Set other values from customizer */
        ResetTimer_WritePeriod(ResetTimer_TC_PERIOD_VALUE );

        #if (ResetTimer__COMPARE == ResetTimer_TC_COMP_CAP_MODE)
            ResetTimer_WriteCompare(ResetTimer_TC_COMPARE_VALUE);

            #if (1u == ResetTimer_TC_COMPARE_SWAP)
                ResetTimer_SetCompareSwap(1u);
                ResetTimer_WriteCompareBuf(ResetTimer_TC_COMPARE_BUF_VALUE);
            #endif  /* (1u == ResetTimer_TC_COMPARE_SWAP) */
        #endif  /* (ResetTimer__COMPARE == ResetTimer_TC_COMP_CAP_MODE) */

        /* Initialize counter value */
        #if (ResetTimer_CY_TCPWM_V2 && ResetTimer_TIMER_UPDOWN_CNT_USED && !ResetTimer_CY_TCPWM_4000)
            ResetTimer_WriteCounter(1u);
        #elif(ResetTimer__COUNT_DOWN == ResetTimer_TC_COUNTER_MODE)
            ResetTimer_WriteCounter(ResetTimer_TC_PERIOD_VALUE);
        #else
            ResetTimer_WriteCounter(0u);
        #endif /* (ResetTimer_CY_TCPWM_V2 && ResetTimer_TIMER_UPDOWN_CNT_USED && !ResetTimer_CY_TCPWM_4000) */
    #endif  /* (ResetTimer__TIMER == ResetTimer_CONFIG) */

    #if (ResetTimer__PWM_SEL == ResetTimer_CONFIG)
        ResetTimer_CONTROL_REG = ResetTimer_CTRL_PWM_BASE_CONFIG;

        #if (ResetTimer__PWM_PR == ResetTimer_PWM_MODE)
            ResetTimer_CONTROL_REG |= ResetTimer_CTRL_PWM_RUN_MODE;
            ResetTimer_WriteCounter(ResetTimer_PWM_PR_INIT_VALUE);
        #else
            ResetTimer_CONTROL_REG |= ResetTimer_CTRL_PWM_ALIGN | ResetTimer_CTRL_PWM_KILL_EVENT;
            
            /* Initialize counter value */
            #if (ResetTimer_CY_TCPWM_V2 && ResetTimer_PWM_UPDOWN_CNT_USED && !ResetTimer_CY_TCPWM_4000)
                ResetTimer_WriteCounter(1u);
            #elif (ResetTimer__RIGHT == ResetTimer_PWM_ALIGN)
                ResetTimer_WriteCounter(ResetTimer_PWM_PERIOD_VALUE);
            #else 
                ResetTimer_WriteCounter(0u);
            #endif  /* (ResetTimer_CY_TCPWM_V2 && ResetTimer_PWM_UPDOWN_CNT_USED && !ResetTimer_CY_TCPWM_4000) */
        #endif  /* (ResetTimer__PWM_PR == ResetTimer_PWM_MODE) */

        #if (ResetTimer__PWM_DT == ResetTimer_PWM_MODE)
            ResetTimer_CONTROL_REG |= ResetTimer_CTRL_PWM_DEAD_TIME_CYCLE;
        #endif  /* (ResetTimer__PWM_DT == ResetTimer_PWM_MODE) */

        #if (ResetTimer__PWM == ResetTimer_PWM_MODE)
            ResetTimer_CONTROL_REG |= ResetTimer_CTRL_PWM_PRESCALER;
        #endif  /* (ResetTimer__PWM == ResetTimer_PWM_MODE) */

        /* Set values from customizer to CTRL1 */
        ResetTimer_TRIG_CONTROL1_REG  = ResetTimer_PWM_SIGNALS_MODES;
    
        /* Set values from customizer to INTR */
        ResetTimer_SetInterruptMode(ResetTimer_PWM_INTERRUPT_MASK);

        /* Set values from customizer to CTRL2 */
        #if (ResetTimer__PWM_PR == ResetTimer_PWM_MODE)
            ResetTimer_TRIG_CONTROL2_REG =
                    (ResetTimer_CC_MATCH_NO_CHANGE    |
                    ResetTimer_OVERLOW_NO_CHANGE      |
                    ResetTimer_UNDERFLOW_NO_CHANGE);
        #else
            #if (ResetTimer__LEFT == ResetTimer_PWM_ALIGN)
                ResetTimer_TRIG_CONTROL2_REG = ResetTimer_PWM_MODE_LEFT;
            #endif  /* ( ResetTimer_PWM_LEFT == ResetTimer_PWM_ALIGN) */

            #if (ResetTimer__RIGHT == ResetTimer_PWM_ALIGN)
                ResetTimer_TRIG_CONTROL2_REG = ResetTimer_PWM_MODE_RIGHT;
            #endif  /* ( ResetTimer_PWM_RIGHT == ResetTimer_PWM_ALIGN) */

            #if (ResetTimer__CENTER == ResetTimer_PWM_ALIGN)
                ResetTimer_TRIG_CONTROL2_REG = ResetTimer_PWM_MODE_CENTER;
            #endif  /* ( ResetTimer_PWM_CENTER == ResetTimer_PWM_ALIGN) */

            #if (ResetTimer__ASYMMETRIC == ResetTimer_PWM_ALIGN)
                ResetTimer_TRIG_CONTROL2_REG = ResetTimer_PWM_MODE_ASYM;
            #endif  /* (ResetTimer__ASYMMETRIC == ResetTimer_PWM_ALIGN) */
        #endif  /* (ResetTimer__PWM_PR == ResetTimer_PWM_MODE) */

        /* Set other values from customizer */
        ResetTimer_WritePeriod(ResetTimer_PWM_PERIOD_VALUE );
        ResetTimer_WriteCompare(ResetTimer_PWM_COMPARE_VALUE);

        #if (1u == ResetTimer_PWM_COMPARE_SWAP)
            ResetTimer_SetCompareSwap(1u);
            ResetTimer_WriteCompareBuf(ResetTimer_PWM_COMPARE_BUF_VALUE);
        #endif  /* (1u == ResetTimer_PWM_COMPARE_SWAP) */

        #if (1u == ResetTimer_PWM_PERIOD_SWAP)
            ResetTimer_SetPeriodSwap(1u);
            ResetTimer_WritePeriodBuf(ResetTimer_PWM_PERIOD_BUF_VALUE);
        #endif  /* (1u == ResetTimer_PWM_PERIOD_SWAP) */
    #endif  /* (ResetTimer__PWM_SEL == ResetTimer_CONFIG) */
    
}


/*******************************************************************************
* Function Name: ResetTimer_Enable
********************************************************************************
*
* Summary:
*  Enables the ResetTimer.
*
* Parameters:
*  None
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_Enable(void)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();
    ResetTimer_BLOCK_CONTROL_REG |= ResetTimer_MASK;
    CyExitCriticalSection(enableInterrupts);

    /* Start Timer or PWM if start input is absent */
    #if (ResetTimer__PWM_SEL == ResetTimer_CONFIG)
        #if (0u == ResetTimer_PWM_START_SIGNAL_PRESENT)
            ResetTimer_TriggerCommand(ResetTimer_MASK, ResetTimer_CMD_START);
        #endif /* (0u == ResetTimer_PWM_START_SIGNAL_PRESENT) */
    #endif /* (ResetTimer__PWM_SEL == ResetTimer_CONFIG) */

    #if (ResetTimer__TIMER == ResetTimer_CONFIG)
        #if (0u == ResetTimer_TC_START_SIGNAL_PRESENT)
            ResetTimer_TriggerCommand(ResetTimer_MASK, ResetTimer_CMD_START);
        #endif /* (0u == ResetTimer_TC_START_SIGNAL_PRESENT) */
    #endif /* (ResetTimer__TIMER == ResetTimer_CONFIG) */
    
    #if (ResetTimer__QUAD == ResetTimer_CONFIG)
        #if (0u != ResetTimer_QUAD_AUTO_START)
            ResetTimer_TriggerCommand(ResetTimer_MASK, ResetTimer_CMD_RELOAD);
        #endif /* (0u != ResetTimer_QUAD_AUTO_START) */
    #endif  /* (ResetTimer__QUAD == ResetTimer_CONFIG) */
}


/*******************************************************************************
* Function Name: ResetTimer_Start
********************************************************************************
*
* Summary:
*  Initializes the ResetTimer with default customizer
*  values when called the first time and enables the ResetTimer.
*  For subsequent calls the configuration is left unchanged and the component is
*  just enabled.
*
* Parameters:
*  None
*
* Return:
*  None
*
* Global variables:
*  ResetTimer_initVar: global variable is used to indicate initial
*  configuration of this component.  The variable is initialized to zero and set
*  to 1 the first time ResetTimer_Start() is called. This allows
*  enabling/disabling a component without re-initialization in all subsequent
*  calls to the ResetTimer_Start() routine.
*
*******************************************************************************/
void ResetTimer_Start(void)
{
    if (0u == ResetTimer_initVar)
    {
        ResetTimer_Init();
        ResetTimer_initVar = 1u;
    }

    ResetTimer_Enable();
}


/*******************************************************************************
* Function Name: ResetTimer_Stop
********************************************************************************
*
* Summary:
*  Disables the ResetTimer.
*
* Parameters:
*  None
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_Stop(void)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_BLOCK_CONTROL_REG &= (uint32)~ResetTimer_MASK;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetMode
********************************************************************************
*
* Summary:
*  Sets the operation mode of the ResetTimer. This function is used when
*  configured as a generic ResetTimer and the actual mode of operation is
*  set at runtime. The mode must be set while the component is disabled.
*
* Parameters:
*  mode: Mode for the ResetTimer to operate in
*   Values:
*   - ResetTimer_MODE_TIMER_COMPARE - Timer / Counter with
*                                                 compare capability
*         - ResetTimer_MODE_TIMER_CAPTURE - Timer / Counter with
*                                                 capture capability
*         - ResetTimer_MODE_QUAD - Quadrature decoder
*         - ResetTimer_MODE_PWM - PWM
*         - ResetTimer_MODE_PWM_DT - PWM with dead time
*         - ResetTimer_MODE_PWM_PR - PWM with pseudo random capability
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetMode(uint32 mode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_MODE_MASK;
    ResetTimer_CONTROL_REG |= mode;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetQDMode
********************************************************************************
*
* Summary:
*  Sets the the Quadrature Decoder to one of the 3 supported modes.
*  Its functionality is only applicable to Quadrature Decoder operation.
*
* Parameters:
*  qdMode: Quadrature Decoder mode
*   Values:
*         - ResetTimer_MODE_X1 - Counts on phi 1 rising
*         - ResetTimer_MODE_X2 - Counts on both edges of phi1 (2x faster)
*         - ResetTimer_MODE_X4 - Counts on both edges of phi1 and phi2
*                                        (4x faster)
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetQDMode(uint32 qdMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_QUAD_MODE_MASK;
    ResetTimer_CONTROL_REG |= qdMode;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetPrescaler
********************************************************************************
*
* Summary:
*  Sets the prescaler value that is applied to the clock input.  Not applicable
*  to a PWM with the dead time mode or Quadrature Decoder mode.
*
* Parameters:
*  prescaler: Prescaler divider value
*   Values:
*         - ResetTimer_PRESCALE_DIVBY1    - Divide by 1 (no prescaling)
*         - ResetTimer_PRESCALE_DIVBY2    - Divide by 2
*         - ResetTimer_PRESCALE_DIVBY4    - Divide by 4
*         - ResetTimer_PRESCALE_DIVBY8    - Divide by 8
*         - ResetTimer_PRESCALE_DIVBY16   - Divide by 16
*         - ResetTimer_PRESCALE_DIVBY32   - Divide by 32
*         - ResetTimer_PRESCALE_DIVBY64   - Divide by 64
*         - ResetTimer_PRESCALE_DIVBY128  - Divide by 128
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetPrescaler(uint32 prescaler)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_PRESCALER_MASK;
    ResetTimer_CONTROL_REG |= prescaler;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetOneShot
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the ResetTimer runs
*  continuously or stops when terminal count is reached.  By default the
*  ResetTimer operates in the continuous mode.
*
* Parameters:
*  oneShotEnable
*   Values:
*     - 0 - Continuous
*     - 1 - Enable One Shot
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetOneShot(uint32 oneShotEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_ONESHOT_MASK;
    ResetTimer_CONTROL_REG |= ((uint32)((oneShotEnable & ResetTimer_1BIT_MASK) <<
                                                               ResetTimer_ONESHOT_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetPWMMode
********************************************************************************
*
* Summary:
*  Writes the control register that determines what mode of operation the PWM
*  output lines are driven in.  There is a setting for what to do on a
*  comparison match (CC_MATCH), on an overflow (OVERFLOW) and on an underflow
*  (UNDERFLOW).  The value for each of the three must be ORed together to form
*  the mode.
*
* Parameters:
*  modeMask: A combination of three mode settings.  Mask must include a value
*  for each of the three or use one of the preconfigured PWM settings.
*   Values:
*     - CC_MATCH_SET        - Set on comparison match
*     - CC_MATCH_CLEAR      - Clear on comparison match
*     - CC_MATCH_INVERT     - Invert on comparison match
*     - CC_MATCH_NO_CHANGE  - No change on comparison match
*     - OVERLOW_SET         - Set on overflow
*     - OVERLOW_CLEAR       - Clear on  overflow
*     - OVERLOW_INVERT      - Invert on overflow
*     - OVERLOW_NO_CHANGE   - No change on overflow
*     - UNDERFLOW_SET       - Set on underflow
*     - UNDERFLOW_CLEAR     - Clear on underflow
*     - UNDERFLOW_INVERT    - Invert on underflow
*     - UNDERFLOW_NO_CHANGE - No change on underflow
*     - PWM_MODE_LEFT       - Setting for left aligned PWM.  Should be combined
*                             with up counting mode
*     - PWM_MODE_RIGHT      - Setting for right aligned PWM.  Should be combined
*                             with down counting mode
*     - PWM_MODE_CENTER     - Setting for center aligned PWM.  Should be
*                             combined with up/down 0 mode
*     - PWM_MODE_ASYM       - Setting for asymmetric PWM.  Should be combined
*                             with up/down 1 mode
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetPWMMode(uint32 modeMask)
{
    ResetTimer_TRIG_CONTROL2_REG = (modeMask & ResetTimer_6BIT_MASK);
}



/*******************************************************************************
* Function Name: ResetTimer_SetPWMSyncKill
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the PWM kill signal (stop input)
*  causes asynchronous or synchronous kill operation.  By default the kill
*  operation is asynchronous.  This functionality is only applicable to the PWM
*  and PWM with dead time modes.
*
*  For Synchronous mode the kill signal disables both the line and line_n
*  signals until the next terminal count.
*
*  For Asynchronous mode the kill signal disables both the line and line_n
*  signals when the kill signal is present.  This mode should only be used
*  when the kill signal (stop input) is configured in the pass through mode
*  (Level sensitive signal).

*
* Parameters:
*  syncKillEnable
*   Values:
*     - 0 - Asynchronous
*     - 1 - Synchronous
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetPWMSyncKill(uint32 syncKillEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_PWM_SYNC_KILL_MASK;
    ResetTimer_CONTROL_REG |= ((uint32)((syncKillEnable & ResetTimer_1BIT_MASK)  <<
                                               ResetTimer_PWM_SYNC_KILL_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetPWMStopOnKill
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the PWM kill signal (stop input)
*  causes the PWM counter to stop.  By default the kill operation does not stop
*  the counter.  This functionality is only applicable to the three PWM modes.
*
*
* Parameters:
*  stopOnKillEnable
*   Values:
*     - 0 - Don't stop
*     - 1 - Stop
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetPWMStopOnKill(uint32 stopOnKillEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_PWM_STOP_KILL_MASK;
    ResetTimer_CONTROL_REG |= ((uint32)((stopOnKillEnable & ResetTimer_1BIT_MASK)  <<
                                                         ResetTimer_PWM_STOP_KILL_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetPWMDeadTime
********************************************************************************
*
* Summary:
*  Writes the dead time control value.  This value delays the rising edge of
*  both the line and line_n signals the designated number of cycles resulting
*  in both signals being inactive for that many cycles.  This functionality is
*  only applicable to the PWM in the dead time mode.

*
* Parameters:
*  Dead time to insert
*   Values: 0 to 255
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetPWMDeadTime(uint32 deadTime)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_PRESCALER_MASK;
    ResetTimer_CONTROL_REG |= ((uint32)((deadTime & ResetTimer_8BIT_MASK) <<
                                                          ResetTimer_PRESCALER_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetPWMInvert
********************************************************************************
*
* Summary:
*  Writes the bits that control whether the line and line_n outputs are
*  inverted from their normal output values.  This functionality is only
*  applicable to the three PWM modes.
*
* Parameters:
*  mask: Mask of outputs to invert.
*   Values:
*         - ResetTimer_INVERT_LINE   - Inverts the line output
*         - ResetTimer_INVERT_LINE_N - Inverts the line_n output
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetPWMInvert(uint32 mask)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_INV_OUT_MASK;
    ResetTimer_CONTROL_REG |= mask;

    CyExitCriticalSection(enableInterrupts);
}



/*******************************************************************************
* Function Name: ResetTimer_WriteCounter
********************************************************************************
*
* Summary:
*  Writes a new 16bit counter value directly into the counter register, thus
*  setting the counter (not the period) to the value written. It is not
*  advised to write to this field when the counter is running.
*
* Parameters:
*  count: value to write
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_WriteCounter(uint32 count)
{
    ResetTimer_COUNTER_REG = (count & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_ReadCounter
********************************************************************************
*
* Summary:
*  Reads the current counter value.
*
* Parameters:
*  None
*
* Return:
*  Current counter value
*
*******************************************************************************/
uint32 ResetTimer_ReadCounter(void)
{
    return (ResetTimer_COUNTER_REG & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_SetCounterMode
********************************************************************************
*
* Summary:
*  Sets the counter mode.  Applicable to all modes except Quadrature Decoder
*  and the PWM with a pseudo random output.
*
* Parameters:
*  counterMode: Enumerated counter type values
*   Values:
*     - ResetTimer_COUNT_UP       - Counts up
*     - ResetTimer_COUNT_DOWN     - Counts down
*     - ResetTimer_COUNT_UPDOWN0  - Counts up and down. Terminal count
*                                         generated when counter reaches 0
*     - ResetTimer_COUNT_UPDOWN1  - Counts up and down. Terminal count
*                                         generated both when counter reaches 0
*                                         and period
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetCounterMode(uint32 counterMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_UPDOWN_MASK;
    ResetTimer_CONTROL_REG |= counterMode;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_WritePeriod
********************************************************************************
*
* Summary:
*  Writes the 16 bit period register with the new period value.
*  To cause the counter to count for N cycles this register should be written
*  with N-1 (counts from 0 to period inclusive).
*
* Parameters:
*  period: Period value
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_WritePeriod(uint32 period)
{
    ResetTimer_PERIOD_REG = (period & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_ReadPeriod
********************************************************************************
*
* Summary:
*  Reads the 16 bit period register.
*
* Parameters:
*  None
*
* Return:
*  Period value
*
*******************************************************************************/
uint32 ResetTimer_ReadPeriod(void)
{
    return (ResetTimer_PERIOD_REG & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_SetCompareSwap
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the compare registers are
*  swapped. When enabled in the Timer/Counter mode(without capture) the swap
*  occurs at a TC event. In the PWM mode the swap occurs at the next TC event
*  following a hardware switch event.
*
* Parameters:
*  swapEnable
*   Values:
*     - 0 - Disable swap
*     - 1 - Enable swap
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetCompareSwap(uint32 swapEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_RELOAD_CC_MASK;
    ResetTimer_CONTROL_REG |= (swapEnable & ResetTimer_1BIT_MASK);

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_WritePeriodBuf
********************************************************************************
*
* Summary:
*  Writes the 16 bit period buf register with the new period value.
*
* Parameters:
*  periodBuf: Period value
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_WritePeriodBuf(uint32 periodBuf)
{
    ResetTimer_PERIOD_BUF_REG = (periodBuf & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_ReadPeriodBuf
********************************************************************************
*
* Summary:
*  Reads the 16 bit period buf register.
*
* Parameters:
*  None
*
* Return:
*  Period value
*
*******************************************************************************/
uint32 ResetTimer_ReadPeriodBuf(void)
{
    return (ResetTimer_PERIOD_BUF_REG & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_SetPeriodSwap
********************************************************************************
*
* Summary:
*  Writes the register that controls whether the period registers are
*  swapped. When enabled in Timer/Counter mode the swap occurs at a TC event.
*  In the PWM mode the swap occurs at the next TC event following a hardware
*  switch event.
*
* Parameters:
*  swapEnable
*   Values:
*     - 0 - Disable swap
*     - 1 - Enable swap
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetPeriodSwap(uint32 swapEnable)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_CONTROL_REG &= (uint32)~ResetTimer_RELOAD_PERIOD_MASK;
    ResetTimer_CONTROL_REG |= ((uint32)((swapEnable & ResetTimer_1BIT_MASK) <<
                                                            ResetTimer_RELOAD_PERIOD_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_WriteCompare
********************************************************************************
*
* Summary:
*  Writes the 16 bit compare register with the new compare value. Not
*  applicable for Timer/Counter with Capture or in Quadrature Decoder modes.
*
* Parameters:
*  compare: Compare value
*
* Return:
*  None
*
* Note:
*  It is not recommended to use the value equal to "0" or equal to 
*  "period value" in Center or Asymmetric align PWM modes on the 
*  PSoC 4100/PSoC 4200 devices.
*  PSoC 4000 devices write the 16 bit compare register with the decremented 
*  compare value in the Up counting mode (except 0x0u), and the incremented 
*  compare value in the Down counting mode (except 0xFFFFu).
*
*******************************************************************************/
void ResetTimer_WriteCompare(uint32 compare)
{
    #if (ResetTimer_CY_TCPWM_4000)
        uint32 currentMode;
    #endif /* (ResetTimer_CY_TCPWM_4000) */

    #if (ResetTimer_CY_TCPWM_4000)
        currentMode = ((ResetTimer_CONTROL_REG & ResetTimer_UPDOWN_MASK) >> ResetTimer_UPDOWN_SHIFT);

        if (((uint32)ResetTimer__COUNT_DOWN == currentMode) && (0xFFFFu != compare))
        {
            compare++;
        }
        else if (((uint32)ResetTimer__COUNT_UP == currentMode) && (0u != compare))
        {
            compare--;
        }
        else
        {
        }
        
    
    #endif /* (ResetTimer_CY_TCPWM_4000) */
    
    ResetTimer_COMP_CAP_REG = (compare & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_ReadCompare
********************************************************************************
*
* Summary:
*  Reads the compare register. Not applicable for Timer/Counter with Capture
*  or in Quadrature Decoder modes.
*  PSoC 4000 devices read the incremented compare register value in the 
*  Up counting mode (except 0xFFFFu), and the decremented value in the 
*  Down counting mode (except 0x0u).
*
* Parameters:
*  None
*
* Return:
*  Compare value
*
* Note:
*  PSoC 4000 devices read the incremented compare register value in the 
*  Up counting mode (except 0xFFFFu), and the decremented value in the 
*  Down counting mode (except 0x0u).
*
*******************************************************************************/
uint32 ResetTimer_ReadCompare(void)
{
    #if (ResetTimer_CY_TCPWM_4000)
        uint32 currentMode;
        uint32 regVal;
    #endif /* (ResetTimer_CY_TCPWM_4000) */

    #if (ResetTimer_CY_TCPWM_4000)
        currentMode = ((ResetTimer_CONTROL_REG & ResetTimer_UPDOWN_MASK) >> ResetTimer_UPDOWN_SHIFT);
        
        regVal = ResetTimer_COMP_CAP_REG;
        
        if (((uint32)ResetTimer__COUNT_DOWN == currentMode) && (0u != regVal))
        {
            regVal--;
        }
        else if (((uint32)ResetTimer__COUNT_UP == currentMode) && (0xFFFFu != regVal))
        {
            regVal++;
        }
        else
        {
        }

        return (regVal & ResetTimer_16BIT_MASK);
    #else
        return (ResetTimer_COMP_CAP_REG & ResetTimer_16BIT_MASK);
    #endif /* (ResetTimer_CY_TCPWM_4000) */
}


/*******************************************************************************
* Function Name: ResetTimer_WriteCompareBuf
********************************************************************************
*
* Summary:
*  Writes the 16 bit compare buffer register with the new compare value. Not
*  applicable for Timer/Counter with Capture or in Quadrature Decoder modes.
*
* Parameters:
*  compareBuf: Compare value
*
* Return:
*  None
*
* Note:
*  It is not recommended to use the value equal to "0" or equal to 
*  "period value" in Center or Asymmetric align PWM modes on the 
*  PSoC 4100/PSoC 4200 devices.
*  PSoC 4000 devices write the 16 bit compare register with the decremented 
*  compare value in the Up counting mode (except 0x0u), and the incremented 
*  compare value in the Down counting mode (except 0xFFFFu).
*
*******************************************************************************/
void ResetTimer_WriteCompareBuf(uint32 compareBuf)
{
    #if (ResetTimer_CY_TCPWM_4000)
        uint32 currentMode;
    #endif /* (ResetTimer_CY_TCPWM_4000) */

    #if (ResetTimer_CY_TCPWM_4000)
        currentMode = ((ResetTimer_CONTROL_REG & ResetTimer_UPDOWN_MASK) >> ResetTimer_UPDOWN_SHIFT);

        if (((uint32)ResetTimer__COUNT_DOWN == currentMode) && (0xFFFFu != compareBuf))
        {
            compareBuf++;
        }
        else if (((uint32)ResetTimer__COUNT_UP == currentMode) && (0u != compareBuf))
        {
            compareBuf --;
        }
        else
        {
        }
    #endif /* (ResetTimer_CY_TCPWM_4000) */
    
    ResetTimer_COMP_CAP_BUF_REG = (compareBuf & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_ReadCompareBuf
********************************************************************************
*
* Summary:
*  Reads the compare buffer register. Not applicable for Timer/Counter with
*  Capture or in Quadrature Decoder modes.
*
* Parameters:
*  None
*
* Return:
*  Compare buffer value
*
* Note:
*  PSoC 4000 devices read the incremented compare register value in the 
*  Up counting mode (except 0xFFFFu), and the decremented value in the 
*  Down counting mode (except 0x0u).
*
*******************************************************************************/
uint32 ResetTimer_ReadCompareBuf(void)
{
    #if (ResetTimer_CY_TCPWM_4000)
        uint32 currentMode;
        uint32 regVal;
    #endif /* (ResetTimer_CY_TCPWM_4000) */

    #if (ResetTimer_CY_TCPWM_4000)
        currentMode = ((ResetTimer_CONTROL_REG & ResetTimer_UPDOWN_MASK) >> ResetTimer_UPDOWN_SHIFT);

        regVal = ResetTimer_COMP_CAP_BUF_REG;
        
        if (((uint32)ResetTimer__COUNT_DOWN == currentMode) && (0u != regVal))
        {
            regVal--;
        }
        else if (((uint32)ResetTimer__COUNT_UP == currentMode) && (0xFFFFu != regVal))
        {
            regVal++;
        }
        else
        {
        }

        return (regVal & ResetTimer_16BIT_MASK);
    #else
        return (ResetTimer_COMP_CAP_BUF_REG & ResetTimer_16BIT_MASK);
    #endif /* (ResetTimer_CY_TCPWM_4000) */
}


/*******************************************************************************
* Function Name: ResetTimer_ReadCapture
********************************************************************************
*
* Summary:
*  Reads the captured counter value. This API is applicable only for
*  Timer/Counter with the capture mode and Quadrature Decoder modes.
*
* Parameters:
*  None
*
* Return:
*  Capture value
*
*******************************************************************************/
uint32 ResetTimer_ReadCapture(void)
{
    return (ResetTimer_COMP_CAP_REG & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_ReadCaptureBuf
********************************************************************************
*
* Summary:
*  Reads the capture buffer register. This API is applicable only for
*  Timer/Counter with the capture mode and Quadrature Decoder modes.
*
* Parameters:
*  None
*
* Return:
*  Capture buffer value
*
*******************************************************************************/
uint32 ResetTimer_ReadCaptureBuf(void)
{
    return (ResetTimer_COMP_CAP_BUF_REG & ResetTimer_16BIT_MASK);
}


/*******************************************************************************
* Function Name: ResetTimer_SetCaptureMode
********************************************************************************
*
* Summary:
*  Sets the capture trigger mode. For PWM mode this is the switch input.
*  This input is not applicable to the Timer/Counter without Capture and
*  Quadrature Decoder modes.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - ResetTimer_TRIG_LEVEL     - Level
*     - ResetTimer_TRIG_RISING    - Rising edge
*     - ResetTimer_TRIG_FALLING   - Falling edge
*     - ResetTimer_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetCaptureMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_TRIG_CONTROL1_REG &= (uint32)~ResetTimer_CAPTURE_MASK;
    ResetTimer_TRIG_CONTROL1_REG |= triggerMode;

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetReloadMode
********************************************************************************
*
* Summary:
*  Sets the reload trigger mode. For Quadrature Decoder mode this is the index
*  input.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - ResetTimer_TRIG_LEVEL     - Level
*     - ResetTimer_TRIG_RISING    - Rising edge
*     - ResetTimer_TRIG_FALLING   - Falling edge
*     - ResetTimer_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetReloadMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_TRIG_CONTROL1_REG &= (uint32)~ResetTimer_RELOAD_MASK;
    ResetTimer_TRIG_CONTROL1_REG |= ((uint32)(triggerMode << ResetTimer_RELOAD_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetStartMode
********************************************************************************
*
* Summary:
*  Sets the start trigger mode. For Quadrature Decoder mode this is the
*  phiB input.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - ResetTimer_TRIG_LEVEL     - Level
*     - ResetTimer_TRIG_RISING    - Rising edge
*     - ResetTimer_TRIG_FALLING   - Falling edge
*     - ResetTimer_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetStartMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_TRIG_CONTROL1_REG &= (uint32)~ResetTimer_START_MASK;
    ResetTimer_TRIG_CONTROL1_REG |= ((uint32)(triggerMode << ResetTimer_START_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetStopMode
********************************************************************************
*
* Summary:
*  Sets the stop trigger mode. For PWM mode this is the kill input.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - ResetTimer_TRIG_LEVEL     - Level
*     - ResetTimer_TRIG_RISING    - Rising edge
*     - ResetTimer_TRIG_FALLING   - Falling edge
*     - ResetTimer_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetStopMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_TRIG_CONTROL1_REG &= (uint32)~ResetTimer_STOP_MASK;
    ResetTimer_TRIG_CONTROL1_REG |= ((uint32)(triggerMode << ResetTimer_STOP_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_SetCountMode
********************************************************************************
*
* Summary:
*  Sets the count trigger mode. For Quadrature Decoder mode this is the phiA
*  input.
*
* Parameters:
*  triggerMode: Enumerated trigger mode value
*   Values:
*     - ResetTimer_TRIG_LEVEL     - Level
*     - ResetTimer_TRIG_RISING    - Rising edge
*     - ResetTimer_TRIG_FALLING   - Falling edge
*     - ResetTimer_TRIG_BOTH      - Both rising and falling edge
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetCountMode(uint32 triggerMode)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_TRIG_CONTROL1_REG &= (uint32)~ResetTimer_COUNT_MASK;
    ResetTimer_TRIG_CONTROL1_REG |= ((uint32)(triggerMode << ResetTimer_COUNT_SHIFT));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_TriggerCommand
********************************************************************************
*
* Summary:
*  Triggers the designated command to occur on the designated TCPWM instances.
*  The mask can be used to apply this command simultaneously to more than one
*  instance.  This allows multiple TCPWM instances to be synchronized.
*
* Parameters:
*  mask: A combination of mask bits for each instance of the TCPWM that the
*        command should apply to.  This function from one instance can be used
*        to apply the command to any of the instances in the design.
*        The mask value for a specific instance is available with the MASK
*        define.
*  command: Enumerated command values. Capture command only applicable for
*           Timer/Counter with Capture and PWM modes.
*   Values:
*     - ResetTimer_CMD_CAPTURE    - Trigger Capture/Switch command
*     - ResetTimer_CMD_RELOAD     - Trigger Reload/Index command
*     - ResetTimer_CMD_STOP       - Trigger Stop/Kill command
*     - ResetTimer_CMD_START      - Trigger Start/phiB command
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_TriggerCommand(uint32 mask, uint32 command)
{
    uint8 enableInterrupts;

    enableInterrupts = CyEnterCriticalSection();

    ResetTimer_COMMAND_REG = ((uint32)(mask << command));

    CyExitCriticalSection(enableInterrupts);
}


/*******************************************************************************
* Function Name: ResetTimer_ReadStatus
********************************************************************************
*
* Summary:
*  Reads the status of the ResetTimer.
*
* Parameters:
*  None
*
* Return:
*  Status
*   Values:
*     - ResetTimer_STATUS_DOWN    - Set if counting down
*     - ResetTimer_STATUS_RUNNING - Set if counter is running
*
*******************************************************************************/
uint32 ResetTimer_ReadStatus(void)
{
    return ((ResetTimer_STATUS_REG >> ResetTimer_RUNNING_STATUS_SHIFT) |
            (ResetTimer_STATUS_REG & ResetTimer_STATUS_DOWN));
}


/*******************************************************************************
* Function Name: ResetTimer_SetInterruptMode
********************************************************************************
*
* Summary:
*  Sets the interrupt mask to control which interrupt
*  requests generate the interrupt signal.
*
* Parameters:
*   interruptMask: Mask of bits to be enabled
*   Values:
*     - ResetTimer_INTR_MASK_TC       - Terminal count mask
*     - ResetTimer_INTR_MASK_CC_MATCH - Compare count / capture mask
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetInterruptMode(uint32 interruptMask)
{
    ResetTimer_INTERRUPT_MASK_REG =  interruptMask;
}


/*******************************************************************************
* Function Name: ResetTimer_GetInterruptSourceMasked
********************************************************************************
*
* Summary:
*  Gets the interrupt requests masked by the interrupt mask.
*
* Parameters:
*   None
*
* Return:
*  Masked interrupt source
*   Values:
*     - ResetTimer_INTR_MASK_TC       - Terminal count mask
*     - ResetTimer_INTR_MASK_CC_MATCH - Compare count / capture mask
*
*******************************************************************************/
uint32 ResetTimer_GetInterruptSourceMasked(void)
{
    return (ResetTimer_INTERRUPT_MASKED_REG);
}


/*******************************************************************************
* Function Name: ResetTimer_GetInterruptSource
********************************************************************************
*
* Summary:
*  Gets the interrupt requests (without masking).
*
* Parameters:
*  None
*
* Return:
*  Interrupt request value
*   Values:
*     - ResetTimer_INTR_MASK_TC       - Terminal count mask
*     - ResetTimer_INTR_MASK_CC_MATCH - Compare count / capture mask
*
*******************************************************************************/
uint32 ResetTimer_GetInterruptSource(void)
{
    return (ResetTimer_INTERRUPT_REQ_REG);
}


/*******************************************************************************
* Function Name: ResetTimer_ClearInterrupt
********************************************************************************
*
* Summary:
*  Clears the interrupt request.
*
* Parameters:
*   interruptMask: Mask of interrupts to clear
*   Values:
*     - ResetTimer_INTR_MASK_TC       - Terminal count mask
*     - ResetTimer_INTR_MASK_CC_MATCH - Compare count / capture mask
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_ClearInterrupt(uint32 interruptMask)
{
    ResetTimer_INTERRUPT_REQ_REG = interruptMask;
}


/*******************************************************************************
* Function Name: ResetTimer_SetInterrupt
********************************************************************************
*
* Summary:
*  Sets a software interrupt request.
*
* Parameters:
*   interruptMask: Mask of interrupts to set
*   Values:
*     - ResetTimer_INTR_MASK_TC       - Terminal count mask
*     - ResetTimer_INTR_MASK_CC_MATCH - Compare count / capture mask
*
* Return:
*  None
*
*******************************************************************************/
void ResetTimer_SetInterrupt(uint32 interruptMask)
{
    ResetTimer_INTERRUPT_SET_REG = interruptMask;
}


/* [] END OF FILE */
