/***************************************************************************//**
* \file SPIM_EZI2C.h
* \version 3.20
*
* \brief
*  This file provides constants and parameter values for the SCB Component in
*  the EZI2C mode.
*
* Note:
*
********************************************************************************
* \copyright
* Copyright 2013-2016, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
*******************************************************************************/

#if !defined(CY_SCB_EZI2C_SPIM_H)
#define CY_SCB_EZI2C_SPIM_H

#include "SPIM.h"


/***************************************
*   Initial Parameter Constants
****************************************/

#define SPIM_EZI2C_CLOCK_STRETCHING         (1u)
#define SPIM_EZI2C_MEDIAN_FILTER_ENABLE     (1u)
#define SPIM_EZI2C_NUMBER_OF_ADDRESSES      (0u)
#define SPIM_EZI2C_PRIMARY_SLAVE_ADDRESS    (8u)
#define SPIM_EZI2C_SECONDARY_SLAVE_ADDRESS  (9u)
#define SPIM_EZI2C_SUB_ADDRESS_SIZE         (0u)
#define SPIM_EZI2C_WAKE_ENABLE              (0u)
#define SPIM_EZI2C_DATA_RATE                (100u)
#define SPIM_EZI2C_SLAVE_ADDRESS_MASK       (254u)
#define SPIM_EZI2C_BYTE_MODE_ENABLE         (0u)


/***************************************
*  Conditional Compilation Parameters
****************************************/

#if(SPIM_SCB_MODE_UNCONFIG_CONST_CFG)

    #define SPIM_SUB_ADDRESS_SIZE16             (0u != SPIM_subAddrSize)
    #define SPIM_SECONDARY_ADDRESS_ENABLE       (0u != SPIM_numberOfAddr)

    #define SPIM_EZI2C_EC_AM_ENABLE         (0u != (SPIM_CTRL_REG & \
                                                                SPIM_CTRL_EC_AM_MODE))
    #define SPIM_EZI2C_SCL_STRETCH_ENABLE   (0u != (SPIM_GetSlaveInterruptMode() & \
                                                                SPIM_INTR_SLAVE_I2C_ADDR_MATCH))
    #define SPIM_EZI2C_SCL_STRETCH_DISABLE       (!SPIM_EZI2C_SCL_STRETCH_ENABLE)

    #define SPIM_SECONDARY_ADDRESS_ENABLE_CONST  (1u)
    #define SPIM_SUB_ADDRESS_SIZE16_CONST        (1u)
    #define SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST  (1u)
    #define SPIM_EZI2C_SCL_STRETCH_DISABLE_CONST (1u)
    #define SPIM_EZI2C_WAKE_ENABLE_CONST         (1u)

    #if (SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1)
        #define SPIM_EZI2C_FIFO_SIZE    (SPIM_FIFO_SIZE)
    #else
        #define SPIM_EZI2C_FIFO_SIZE    (SPIM_GET_FIFO_SIZE(SPIM_CTRL_REG & \
                                                                                    SPIM_CTRL_BYTE_MODE))
    #endif /* (SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1) */

#else

    #define SPIM_SUB_ADDRESS_SIZE16         (0u != SPIM_EZI2C_SUB_ADDRESS_SIZE)
    #define SPIM_SUB_ADDRESS_SIZE16_CONST   (SPIM_SUB_ADDRESS_SIZE16)

    #define SPIM_SECONDARY_ADDRESS_ENABLE        (0u != SPIM_EZI2C_NUMBER_OF_ADDRESSES)
    #define SPIM_SECONDARY_ADDRESS_ENABLE_CONST  (SPIM_SECONDARY_ADDRESS_ENABLE)

    #define SPIM_EZI2C_SCL_STRETCH_ENABLE        (0u != SPIM_EZI2C_CLOCK_STRETCHING)
    #define SPIM_EZI2C_SCL_STRETCH_DISABLE       (!SPIM_EZI2C_SCL_STRETCH_ENABLE)
    #define SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST  (SPIM_EZI2C_SCL_STRETCH_ENABLE)
    #define SPIM_EZI2C_SCL_STRETCH_DISABLE_CONST (SPIM_EZI2C_SCL_STRETCH_DISABLE)

    #define SPIM_EZI2C_WAKE_ENABLE_CONST         (0u != SPIM_EZI2C_WAKE_ENABLE)
    #define SPIM_EZI2C_EC_AM_ENABLE              (0u != SPIM_EZI2C_WAKE_ENABLE)

    #define SPIM_EZI2C_FIFO_SIZE SPIM_GET_FIFO_SIZE(SPIM_EZI2C_BYTE_MODE_ENABLE)

#endif /* (SPIM_SCB_MODE_UNCONFIG_CONST_CFG) */


/***************************************
*       Type Definitions
***************************************/
/**
* \addtogroup group_structures
* @{
*/
typedef struct
{
    /** When enabled the SCL is stretched as required for proper operation: 
     *  0  disable, 1  enable.
    */
    uint32 enableClockStretch;
    
    /** This field is left for compatibility and its value is ignored. 
     *  Median filter is disabled for EZI2C mode.
    */
    uint32 enableMedianFilter;
    
    /** Number of supported addresses: 
     *  - SPIM_EZI2C_ONE_ADDRESS
     *  - SPIM_EZI2C_TWO_ADDRESSES
    */
    uint32 numberOfAddresses;
    
    /** Primary 7-bit slave address.
    */
    uint32 primarySlaveAddr;
    
    /** Secondary 7-bit slave address.
    */
    uint32 secondarySlaveAddr;
    
    /** Size of sub-address.
     *  - SPIM_EZI2C_SUB_ADDR8_BITS 
     *  - SPIM_EZI2C_SUB_ADDR16_BITS
    */
    uint32 subAddrSize;
    
    /** When enabled the TX and RX FIFO depth is doubled and equal to 
     *  16 bytes: 0  disable, 1  enable.
    */
    uint32 enableWake;
    
    /** When enabled the TX and RX FIFO depth is 16 bytes: 0  disable, 
     *  1  enable.
     * 
     *  Ignored for all devices other than PSoC 4100 BLE / PSoC 4200 BLE / 
     *  PSoC 4100M / PSoC 4200M / PSoC 4200L / PSoC 4000S / PSoC 4100S / 
     *  PSoC Analog Coprocessor.
    */
    uint8  enableByteMode;
} SPIM_EZI2C_INIT_STRUCT;
/** @} structures */

/***************************************
*        Function Prototypes
***************************************/

/**
* \addtogroup group_ezi2c
* @{
*/

#if(SPIM_SCB_MODE_UNCONFIG_CONST_CFG)
    void SPIM_EzI2CInit(const SPIM_EZI2C_INIT_STRUCT *config);
#endif /* (SPIM_SCB_MODE_UNCONFIG_CONST_CFG) */

uint32 SPIM_EzI2CGetActivity(void);

void   SPIM_EzI2CSetAddress1(uint32 address);
uint32 SPIM_EzI2CGetAddress1(void);
void   SPIM_EzI2CSetBuffer1(uint32 bufSize, uint32 rwBoundary, volatile uint8 * buffer);
void   SPIM_EzI2CSetReadBoundaryBuffer1(uint32 rwBoundary);

#if(SPIM_SECONDARY_ADDRESS_ENABLE_CONST)
    void   SPIM_EzI2CSetAddress2(uint32 address);
    uint32 SPIM_EzI2CGetAddress2(void);
    void   SPIM_EzI2CSetBuffer2(uint32 bufSize, uint32 rwBoundary, volatile uint8 * buffer);
    void   SPIM_EzI2CSetReadBoundaryBuffer2(uint32 rwBoundary);
#endif /* (SPIM_SECONDARY_ADDRESS_ENABLE_CONST) */
/** @} ezi2c */

#if(SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST)
    CY_ISR_PROTO(SPIM_EZI2C_STRETCH_ISR);
#endif /* (SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST) */

#if(SPIM_EZI2C_SCL_STRETCH_DISABLE_CONST)
    CY_ISR_PROTO(SPIM_EZI2C_NO_STRETCH_ISR);
#endif /* (SPIM_EZI2C_SCL_STRETCH_DISABLE) */


/***************************************
*            API Constants
***************************************/

/* Configuration structure constants */
#define SPIM_EZI2C_ONE_ADDRESS      (0u)
#define SPIM_EZI2C_TWO_ADDRESSES    (1u)

#define SPIM_EZI2C_PRIMARY_ADDRESS  (0u)
#define SPIM_EZI2C_BOTH_ADDRESSES   (1u)

#define SPIM_EZI2C_SUB_ADDR8_BITS   (0u)
#define SPIM_EZI2C_SUB_ADDR16_BITS  (1u)

/* SPIM_EzI2CGetActivity() return constants */
#define SPIM_EZI2C_STATUS_SECOND_OFFSET (1u)
#define SPIM_EZI2C_STATUS_READ1     ((uint32) (SPIM_INTR_SLAVE_I2C_NACK))        /* Bit [1]   */
#define SPIM_EZI2C_STATUS_WRITE1    ((uint32) (SPIM_INTR_SLAVE_I2C_WRITE_STOP))  /* Bit [3]   */
#define SPIM_EZI2C_STATUS_READ2     ((uint32) (SPIM_INTR_SLAVE_I2C_NACK >> \
                                                           SPIM_EZI2C_STATUS_SECOND_OFFSET)) /* Bit [0]   */
#define SPIM_EZI2C_STATUS_WRITE2    ((uint32) (SPIM_INTR_SLAVE_I2C_WRITE_STOP >> \
                                                           SPIM_EZI2C_STATUS_SECOND_OFFSET)) /* Bit [2]   */
#define SPIM_EZI2C_STATUS_ERR       ((uint32) (0x10u))                                       /* Bit [4]   */
#define SPIM_EZI2C_STATUS_BUSY      ((uint32) (0x20u))                                       /* Bit [5]   */
#define SPIM_EZI2C_CLEAR_STATUS     ((uint32) (0x1Fu))                                       /* Bits [0-4]*/
#define SPIM_EZI2C_CMPLT_INTR_MASK  ((uint32) (SPIM_INTR_SLAVE_I2C_NACK | \
                                                           SPIM_INTR_SLAVE_I2C_WRITE_STOP))


/***************************************
*     Vars with External Linkage
***************************************/

#if(SPIM_SCB_MODE_UNCONFIG_CONST_CFG)
    extern const SPIM_EZI2C_INIT_STRUCT SPIM_configEzI2C;
#endif /* (SPIM_SCB_MODE_UNCONFIG_CONST_CFG) */


/***************************************
*           FSM states
***************************************/

/* Returns to master when there is no data to transmit */
#define SPIM_EZI2C_OVFL_RETURN  (0xFFu)

/* States of EZI2C Slave FSM */
#define SPIM_EZI2C_FSM_OFFSET_HI8 ((uint8) (0x02u)) /* Idle state for 1 addr: waits for high byte offset */
#define SPIM_EZI2C_FSM_OFFSET_LO8 ((uint8) (0x00u)) /* Idle state for 2 addr: waits for low byte offset  */
#define SPIM_EZI2C_FSM_BYTE_WRITE ((uint8) (0x01u)) /* Data byte write state: byte by byte mode          */
#define SPIM_EZI2C_FSM_WAIT_STOP  ((uint8) (0x03u)) /* Discards written bytes as they do not match write
                                                                   criteria */
#define SPIM_EZI2C_FSM_WRITE_MASK ((uint8) (0x01u)) /* Incorporates write states: EZI2C_FSM_BYTE_WRITE and
                                                                   EZI2C_FSM_WAIT_STOP  */

#define SPIM_EZI2C_FSM_IDLE     ((SPIM_SUB_ADDRESS_SIZE16) ?      \
                                                (SPIM_EZI2C_FSM_OFFSET_HI8) : \
                                                (SPIM_EZI2C_FSM_OFFSET_LO8))

#define SPIM_EZI2C_CONTINUE_TRANSFER    (1u)
#define SPIM_EZI2C_COMPLETE_TRANSFER    (0u)

#define SPIM_EZI2C_NACK_RECEIVED_ADDRESS  (0u)
#define SPIM_EZI2C_ACK_RECEIVED_ADDRESS   (1u)

#define SPIM_EZI2C_ACTIVE_ADDRESS1  (0u)
#define SPIM_EZI2C_ACTIVE_ADDRESS2  (1u)


/***************************************
*       Macro Definitions
***************************************/

/* Access to global variables */
#if(SPIM_SECONDARY_ADDRESS_ENABLE_CONST)

    #define SPIM_EZI2C_UPDATE_LOC_STATUS(activeAddress, locStatus) \
            do{ \
                (locStatus) >>= (activeAddress); \
            }while(0)

    #define SPIM_EZI2C_GET_INDEX(activeAddress) \
            ((SPIM_EZI2C_ACTIVE_ADDRESS1 == (activeAddress)) ? \
                ((uint32) SPIM_indexBuf1) :                    \
                ((uint32) SPIM_indexBuf2))

    #define SPIM_EZI2C_GET_OFFSET(activeAddress) \
            ((SPIM_EZI2C_ACTIVE_ADDRESS1 == (activeAddress)) ? \
                ((uint32) SPIM_offsetBuf1) :                   \
                ((uint32) SPIM_offsetBuf2))

    #define SPIM_EZI2C_SET_INDEX(activeAddress, locIndex) \
            do{ \
                if(SPIM_EZI2C_ACTIVE_ADDRESS1 == (activeAddress)) \
                {    \
                    SPIM_indexBuf1 = (uint16) (locIndex); \
                }    \
                else \
                {    \
                    SPIM_indexBuf2 = (uint16) (locIndex); \
                }    \
            }while(0)

    #define SPIM_EZI2C_SET_OFFSET(activeAddress, locOffset) \
            do{ \
                if(SPIM_EZI2C_ACTIVE_ADDRESS1 == (activeAddress)) \
                {    \
                    SPIM_offsetBuf1 = (uint16) (locOffset); \
                }    \
                else \
                {    \
                    SPIM_offsetBuf2 = (uint16) (locOffset); \
                }    \
            }while(0)
#else
    #define SPIM_EZI2C_UPDATE_LOC_STATUS(activeAddress, locStatus)  do{ /* Empty*/ }while(0)

    #define SPIM_EZI2C_GET_INDEX(activeAddress)  ((uint32) (SPIM_indexBuf1))

    #define SPIM_EZI2C_GET_OFFSET(activeAddress) ((uint32) (SPIM_offsetBuf1))

    #define SPIM_EZI2C_SET_INDEX(activeAddress, locIndex) \
            do{ \
                SPIM_indexBuf1 = (uint16) (locIndex); \
            }while(0)

    #define SPIM_EZI2C_SET_OFFSET(activeAddress, locOffset) \
            do{ \
                SPIM_offsetBuf1 = (uint16) (locOffset); \
            }while(0)

#endif  /* (SPIM_SUB_ADDRESS_SIZE16_CONST) */


/* This macro only applicable for EZI2C slave without clock stretching.
* It should not be used for other pusposes. */
#define SPIM_EZI2C_TX_FIFO_CTRL_SET   (SPIM_EZI2C_TX_FIFO_CTRL | \
                                                   SPIM_TX_FIFO_CTRL_CLEAR)
#define SPIM_EZI2C_TX_FIFO_CTRL_CLEAR (SPIM_EZI2C_TX_FIFO_CTRL)
#define SPIM_FAST_CLEAR_TX_FIFO \
                            do{             \
                                SPIM_TX_FIFO_CTRL_REG = SPIM_EZI2C_TX_FIFO_CTRL_SET;   \
                                SPIM_TX_FIFO_CTRL_REG = SPIM_EZI2C_TX_FIFO_CTRL_CLEAR; \
                            }while(0)


/***************************************
*      Common Register Settings
***************************************/

#define SPIM_CTRL_EZI2C     (SPIM_CTRL_MODE_I2C)

#define SPIM_EZI2C_CTRL     (SPIM_I2C_CTRL_S_GENERAL_IGNORE | \
                                         SPIM_I2C_CTRL_SLAVE_MODE)

#define SPIM_EZI2C_CTRL_AUTO    (SPIM_I2C_CTRL_S_READY_ADDR_ACK      | \
                                             SPIM_I2C_CTRL_S_READY_DATA_ACK      | \
                                             SPIM_I2C_CTRL_S_NOT_READY_ADDR_NACK | \
                                             SPIM_I2C_CTRL_S_NOT_READY_DATA_NACK)

#define SPIM_EZI2C_RX_CTRL  ((SPIM_FIFO_SIZE - 1u)   | \
                                          SPIM_RX_CTRL_MSB_FIRST | \
                                          SPIM_RX_CTRL_ENABLED)

#define SPIM_EZI2C_TX_FIFO_CTRL (2u)
#define SPIM_TX_LOAD_SIZE       (2u)

#define SPIM_EZI2C_TX_CTRL  ((SPIM_ONE_BYTE_WIDTH - 1u) | \
                                          SPIM_TX_CTRL_MSB_FIRST    | \
                                          SPIM_TX_CTRL_ENABLED)

#define SPIM_EZI2C_INTR_SLAVE_MASK  (SPIM_INTR_SLAVE_I2C_BUS_ERROR | \
                                                 SPIM_INTR_SLAVE_I2C_ARB_LOST  | \
                                                 SPIM_INTR_SLAVE_I2C_STOP)

#define SPIM_INTR_SLAVE_COMPLETE    (SPIM_INTR_SLAVE_I2C_STOP | \
                                                 SPIM_INTR_SLAVE_I2C_NACK | \
                                                 SPIM_INTR_SLAVE_I2C_WRITE_STOP)


/***************************************
*    Initialization Register Settings
***************************************/

#if(SPIM_SCB_MODE_EZI2C_CONST_CFG)

    #define SPIM_EZI2C_DEFAULT_CTRL \
                                (SPIM_GET_CTRL_BYTE_MODE  (SPIM_EZI2C_BYTE_MODE_ENABLE)    | \
                                 SPIM_GET_CTRL_ADDR_ACCEPT(SPIM_EZI2C_NUMBER_OF_ADDRESSES) | \
                                 SPIM_GET_CTRL_EC_AM_MODE (SPIM_EZI2C_WAKE_ENABLE))

    #if(SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST)
        #define SPIM_EZI2C_DEFAULT_I2C_CTRL (SPIM_EZI2C_CTRL)
    #else
        #define SPIM_EZI2C_DEFAULT_I2C_CTRL (SPIM_EZI2C_CTRL_AUTO | SPIM_EZI2C_CTRL)
    #endif /* (SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST) */


    #define SPIM_EZI2C_DEFAULT_RX_MATCH \
                                (SPIM_GET_I2C_8BIT_ADDRESS(SPIM_EZI2C_PRIMARY_SLAVE_ADDRESS) | \
                                 SPIM_GET_RX_MATCH_MASK   (SPIM_EZI2C_SLAVE_ADDRESS_MASK))

    #define SPIM_EZI2C_DEFAULT_RX_CTRL  (SPIM_EZI2C_RX_CTRL)
    #define SPIM_EZI2C_DEFAULT_TX_CTRL  (SPIM_EZI2C_TX_CTRL)

    #define SPIM_EZI2C_DEFAULT_RX_FIFO_CTRL (0u)
    #if(SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST)
        #define SPIM_EZI2C_DEFAULT_TX_FIFO_CTRL (0u)
    #else
        #define SPIM_EZI2C_DEFAULT_TX_FIFO_CTRL (2u)
    #endif /* (SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST) */

    /* Interrupt sources */
    #define SPIM_EZI2C_DEFAULT_INTR_I2C_EC_MASK (SPIM_NO_INTR_SOURCES)
    #define SPIM_EZI2C_DEFAULT_INTR_SPI_EC_MASK (SPIM_NO_INTR_SOURCES)
    #define SPIM_EZI2C_DEFAULT_INTR_MASTER_MASK (SPIM_NO_INTR_SOURCES)
    #define SPIM_EZI2C_DEFAULT_INTR_TX_MASK     (SPIM_NO_INTR_SOURCES)

    #if(SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST)
        #define SPIM_EZI2C_DEFAULT_INTR_RX_MASK     (SPIM_NO_INTR_SOURCES)
        #define SPIM_EZI2C_DEFAULT_INTR_SLAVE_MASK  (SPIM_INTR_SLAVE_I2C_ADDR_MATCH | \
                                                                 SPIM_EZI2C_INTR_SLAVE_MASK)
    #else
        #define SPIM_EZI2C_DEFAULT_INTR_RX_MASK     (SPIM_INTR_RX_NOT_EMPTY)
        #define SPIM_EZI2C_DEFAULT_INTR_SLAVE_MASK  (SPIM_INTR_SLAVE_I2C_START | \
                                                                 SPIM_EZI2C_INTR_SLAVE_MASK)
    #endif /* (SPIM_EZI2C_SCL_STRETCH_ENABLE_CONST) */

#endif /* (SPIM_SCB_MODE_EZI2C_CONST_CFG) */

#endif /* (CY_SCB_EZI2C_SPIM_H) */


/* [] END OF FILE */
