/*******************************************************************************
* File Name: Accel_Int.c  
* Version 2.20
*
* Description:
*  This file contains APIs to set up the Pins component for low power modes.
*
* Note:
*
********************************************************************************
* Copyright 2015, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "cytypes.h"
#include "Accel_Int.h"

static Accel_Int_BACKUP_STRUCT  Accel_Int_backup = {0u, 0u, 0u};


/*******************************************************************************
* Function Name: Accel_Int_Sleep
****************************************************************************//**
*
* \brief Stores the pin configuration and prepares the pin for entering chip 
*  deep-sleep/hibernate modes. This function applies only to SIO and USBIO pins.
*  It should not be called for GPIO or GPIO_OVT pins.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None 
*  
* \sideeffect
*  For SIO pins, this function configures the pin input threshold to CMOS and
*  drive level to Vddio. This is needed for SIO pins when in device 
*  deep-sleep/hibernate modes.
*
* \funcusage
*  \snippet Accel_Int_SUT.c usage_Accel_Int_Sleep_Wakeup
*******************************************************************************/
void Accel_Int_Sleep(void)
{
    #if defined(Accel_Int__PC)
        Accel_Int_backup.pcState = Accel_Int_PC;
    #else
        #if (CY_PSOC4_4200L)
            /* Save the regulator state and put the PHY into suspend mode */
            Accel_Int_backup.usbState = Accel_Int_CR1_REG;
            Accel_Int_USB_POWER_REG |= Accel_Int_USBIO_ENTER_SLEEP;
            Accel_Int_CR1_REG &= Accel_Int_USBIO_CR1_OFF;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(Accel_Int__SIO)
        Accel_Int_backup.sioState = Accel_Int_SIO_REG;
        /* SIO requires unregulated output buffer and single ended input buffer */
        Accel_Int_SIO_REG &= (uint32)(~Accel_Int_SIO_LPM_MASK);
    #endif  
}


/*******************************************************************************
* Function Name: Accel_Int_Wakeup
****************************************************************************//**
*
* \brief Restores the pin configuration that was saved during Pin_Sleep(). This 
* function applies only to SIO and USBIO pins. It should not be called for
* GPIO or GPIO_OVT pins.
*
* For USBIO pins, the wakeup is only triggered for falling edge interrupts.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None
*  
* \funcusage
*  Refer to Accel_Int_Sleep() for an example usage.
*******************************************************************************/
void Accel_Int_Wakeup(void)
{
    #if defined(Accel_Int__PC)
        Accel_Int_PC = Accel_Int_backup.pcState;
    #else
        #if (CY_PSOC4_4200L)
            /* Restore the regulator state and come out of suspend mode */
            Accel_Int_USB_POWER_REG &= Accel_Int_USBIO_EXIT_SLEEP_PH1;
            Accel_Int_CR1_REG = Accel_Int_backup.usbState;
            Accel_Int_USB_POWER_REG &= Accel_Int_USBIO_EXIT_SLEEP_PH2;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(Accel_Int__SIO)
        Accel_Int_SIO_REG = Accel_Int_backup.sioState;
    #endif
}


/* [] END OF FILE */
