/*******************************************************************************
* File Name: Kill_Pwr.c  
* Version 2.20
*
* Description:
*  This file contains APIs to set up the Pins component for low power modes.
*
* Note:
*
********************************************************************************
* Copyright 2015, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "cytypes.h"
#include "Kill_Pwr.h"

static Kill_Pwr_BACKUP_STRUCT  Kill_Pwr_backup = {0u, 0u, 0u};


/*******************************************************************************
* Function Name: Kill_Pwr_Sleep
****************************************************************************//**
*
* \brief Stores the pin configuration and prepares the pin for entering chip 
*  deep-sleep/hibernate modes. This function applies only to SIO and USBIO pins.
*  It should not be called for GPIO or GPIO_OVT pins.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None 
*  
* \sideeffect
*  For SIO pins, this function configures the pin input threshold to CMOS and
*  drive level to Vddio. This is needed for SIO pins when in device 
*  deep-sleep/hibernate modes.
*
* \funcusage
*  \snippet Kill_Pwr_SUT.c usage_Kill_Pwr_Sleep_Wakeup
*******************************************************************************/
void Kill_Pwr_Sleep(void)
{
    #if defined(Kill_Pwr__PC)
        Kill_Pwr_backup.pcState = Kill_Pwr_PC;
    #else
        #if (CY_PSOC4_4200L)
            /* Save the regulator state and put the PHY into suspend mode */
            Kill_Pwr_backup.usbState = Kill_Pwr_CR1_REG;
            Kill_Pwr_USB_POWER_REG |= Kill_Pwr_USBIO_ENTER_SLEEP;
            Kill_Pwr_CR1_REG &= Kill_Pwr_USBIO_CR1_OFF;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(Kill_Pwr__SIO)
        Kill_Pwr_backup.sioState = Kill_Pwr_SIO_REG;
        /* SIO requires unregulated output buffer and single ended input buffer */
        Kill_Pwr_SIO_REG &= (uint32)(~Kill_Pwr_SIO_LPM_MASK);
    #endif  
}


/*******************************************************************************
* Function Name: Kill_Pwr_Wakeup
****************************************************************************//**
*
* \brief Restores the pin configuration that was saved during Pin_Sleep(). This 
* function applies only to SIO and USBIO pins. It should not be called for
* GPIO or GPIO_OVT pins.
*
* For USBIO pins, the wakeup is only triggered for falling edge interrupts.
*
* <b>Note</b> This function is available in PSoC 4 only.
*
* \return 
*  None
*  
* \funcusage
*  Refer to Kill_Pwr_Sleep() for an example usage.
*******************************************************************************/
void Kill_Pwr_Wakeup(void)
{
    #if defined(Kill_Pwr__PC)
        Kill_Pwr_PC = Kill_Pwr_backup.pcState;
    #else
        #if (CY_PSOC4_4200L)
            /* Restore the regulator state and come out of suspend mode */
            Kill_Pwr_USB_POWER_REG &= Kill_Pwr_USBIO_EXIT_SLEEP_PH1;
            Kill_Pwr_CR1_REG = Kill_Pwr_backup.usbState;
            Kill_Pwr_USB_POWER_REG &= Kill_Pwr_USBIO_EXIT_SLEEP_PH2;
        #endif
    #endif
    #if defined(CYIPBLOCK_m0s8ioss_VERSION) && defined(Kill_Pwr__SIO)
        Kill_Pwr_SIO_REG = Kill_Pwr_backup.sioState;
    #endif
}


/* [] END OF FILE */
