/***************************************************************************//**
* \file cy_sysanalog.c
* \version 1.0
*
* Provides the public functions for the API for the SAR driver.
*
********************************************************************************
* \copyright
* Copyright 2017, Cypress Semiconductor Corporation. All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
*******************************************************************************/
#include "cy_sysanalog.h"

#if defined(__cplusplus)
extern "C" {
#endif

/* Configure the AREF to use the local Vref and local IZTAT. Can be used with \ref Cy_SysAnalog_Init. */
const cy_stc_sysanalog_config_t Cy_SysAnalog_Fast_Local =
{
    /*.startup    */ CY_SYSANALOG_STARTUP_FAST,
    /*.iztat      */ CY_SYSANALOG_IZTAT_SOURCE_LOCAL,
    /*.vref       */ CY_SYSANALOG_VREF_SOURCE_LOCAL_1_2V,
    /*.deepSleep  */ CY_SYSANALOG_DEEPSLEEP_DISABLE,
    /*.enable     */ true,
};

/* Configure the AREF to use the SRSS Vref and SRSS IZTAT. Can be used with \ref Cy_SysAnalog_Init. */
const cy_stc_sysanalog_config_t Cy_SysAnalog_Fast_SRSS =
{
    /*.startup    */ CY_SYSANALOG_STARTUP_FAST,
    /*.iztat      */ CY_SYSANALOG_IZTAT_SOURCE_SRSS,
    /*.vref       */ CY_SYSANALOG_VREF_SOURCE_SRSS,
    /*.deepSleep  */ CY_SYSANALOG_DEEPSLEEP_DISABLE,
    /*.enable     */ true,
};

/* Configure the AREF to use the external Vref and local IZTAT. Can be used with \ref Cy_SysAnalog_Init. */
const cy_stc_sysanalog_config_t Cy_SysAnalog_Fast_External =
{
    /*.startup    */ CY_SYSANALOG_STARTUP_FAST,
    /*.iztat      */ CY_SYSANALOG_IZTAT_SOURCE_LOCAL,
    /*.vref       */ CY_SYSANALOG_VREF_SOURCE_EXTERNAL,
    /*.deepSleep  */ CY_SYSANALOG_DEEPSLEEP_DISABLE,
    /*.enable     */ true,
};

/*******************************************************************************
* Function Name: Cy_SysAnalog_Init
****************************************************************************//**
*
* Initialize the AREF configuration register.
*
* \param config
* Pointer to structure containing configuration data. See \ref cy_stc_sysanalog_config_t
*
* \return cy_en_sysanalog_status_t
* - CY_SYSANALOG_SUCCESS: initialization complete
* - CY_SYSANALOG_BAD_PARAM: input pointers are null, initialization incomplete
*
*******************************************************************************/
cy_en_sysanalog_status_t Cy_SysAnalog_Init(const cy_stc_sysanalog_config_t *config)
{
    CY_ASSERT_L1(NULL != config);

    cy_en_sysanalog_status_t result;
    uint32_t ctrlReg = CY_SYSANALOG_DEINIT;

    if (NULL == config)
    {
        result = CY_SYSANALOG_BAD_PARAM;
    }
    else
    {
        CY_ASSERT_L3(CY_SYSANALOG_STARTUP(config->startup));
        CY_ASSERT_L3(CY_SYSANALOG_DEEPSLEEP(config->deepSleep));
        CY_ASSERT_L3(CY_SYSANALOG_VREF(config->vref));
        CY_ASSERT_L3(CY_SYSANALOG_IZTAT(config->iztat));

        ctrlReg = (uint32_t) config->startup \
                  | CY_SYSANALOG_DEFAULT_BIAS_SCALE \
                  | (uint32_t) config->iztat \
                  | (uint32_t) config->vref \
                  | (uint32_t) config->deepSleep \
                  | ((uint32_t) config->enable << PASS_AREF_AREF_CTRL_ENABLED_Pos);

        PASS_AREF->AREF_CTRL = ctrlReg;

        result = CY_SYSANALOG_SUCCESS;
    }

    return result;
}

#if defined(__cplusplus)
}
#endif

/* [] END OF FILE */

