/*******************************************************************************
* File Name: PinLED.c  
* Version 2.10
*
* Description:
*  This file contains API to enable firmware control of a Pins component.
*
* Note:
*
********************************************************************************
* Copyright 2008-2014, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "cytypes.h"
#include "PinLED.h"

#define SetP4PinDriveMode(shift, mode)  \
    do { \
        PinLED_PC =   (PinLED_PC & \
                                (uint32)(~(uint32)(PinLED_DRIVE_MODE_IND_MASK << (PinLED_DRIVE_MODE_BITS * (shift))))) | \
                                (uint32)((uint32)(mode) << (PinLED_DRIVE_MODE_BITS * (shift))); \
    } while (0)


/*******************************************************************************
* Function Name: PinLED_Write
********************************************************************************
*
* Summary:
*  Assign a new value to the digital port's data output register.  
*
* Parameters:  
*  prtValue:  The value to be assigned to the Digital Port. 
*
* Return: 
*  None 
*  
*******************************************************************************/
void PinLED_Write(uint8 value) 
{
    uint8 drVal = (uint8)(PinLED_DR & (uint8)(~PinLED_MASK));
    drVal = (drVal | ((uint8)(value << PinLED_SHIFT) & PinLED_MASK));
    PinLED_DR = (uint32)drVal;
}


/*******************************************************************************
* Function Name: PinLED_SetDriveMode
********************************************************************************
*
* Summary:
*  Change the drive mode on the pins of the port.
* 
* Parameters:  
*  mode:  Change the pins to one of the following drive modes.
*
*  PinLED_DM_STRONG     Strong Drive 
*  PinLED_DM_OD_HI      Open Drain, Drives High 
*  PinLED_DM_OD_LO      Open Drain, Drives Low 
*  PinLED_DM_RES_UP     Resistive Pull Up 
*  PinLED_DM_RES_DWN    Resistive Pull Down 
*  PinLED_DM_RES_UPDWN  Resistive Pull Up/Down 
*  PinLED_DM_DIG_HIZ    High Impedance Digital 
*  PinLED_DM_ALG_HIZ    High Impedance Analog 
*
* Return: 
*  None
*
*******************************************************************************/
void PinLED_SetDriveMode(uint8 mode) 
{
	SetP4PinDriveMode(PinLED__0__SHIFT, mode);
}


/*******************************************************************************
* Function Name: PinLED_Read
********************************************************************************
*
* Summary:
*  Read the current value on the pins of the Digital Port in right justified 
*  form.
*
* Parameters:  
*  None 
*
* Return: 
*  Returns the current value of the Digital Port as a right justified number
*  
* Note:
*  Macro PinLED_ReadPS calls this function. 
*  
*******************************************************************************/
uint8 PinLED_Read(void) 
{
    return (uint8)((PinLED_PS & PinLED_MASK) >> PinLED_SHIFT);
}


/*******************************************************************************
* Function Name: PinLED_ReadDataReg
********************************************************************************
*
* Summary:
*  Read the current value assigned to a Digital Port's data output register
*
* Parameters:  
*  None 
*
* Return: 
*  Returns the current value assigned to the Digital Port's data output register
*  
*******************************************************************************/
uint8 PinLED_ReadDataReg(void) 
{
    return (uint8)((PinLED_DR & PinLED_MASK) >> PinLED_SHIFT);
}


/* If Interrupts Are Enabled for this Pins component */ 
#if defined(PinLED_INTSTAT) 

    /*******************************************************************************
    * Function Name: PinLED_ClearInterrupt
    ********************************************************************************
    *
    * Summary:
    *  Clears any active interrupts attached to port and returns the value of the 
    *  interrupt status register.
    *
    * Parameters:  
    *  None 
    *
    * Return: 
    *  Returns the value of the interrupt status register
    *  
    *******************************************************************************/
    uint8 PinLED_ClearInterrupt(void) 
    {
		uint8 maskedStatus = (uint8)(PinLED_INTSTAT & PinLED_MASK);
		PinLED_INTSTAT = maskedStatus;
        return maskedStatus >> PinLED_SHIFT;
    }

#endif /* If Interrupts Are Enabled for this Pins component */ 


/* [] END OF FILE */
