/*******************************************************************************
* File Name: Loopback.c
* Version 0.2
*
*  Description:
*    API for USBFS Component.
*
*   Note:
*   
*   Many of the functions use endpoint number.  RAM arrays are sized with 9
*   elements so they are indexed directly by bEPNumber.  The SIE and ARB
*   registers are indexed by variations of bEPNumber - 1.
*
********************************************************************************
* Copyright (c) 2009, Cypress Semiconductor Corporation.
********************************************************************************
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express, written permission of Cypress.
*******************************************************************************/

#include "device.h"
#include "cyfitter.h"
#include "Loopback.h"

/*******************************************************************************
* External data references
********************************************************************************/
extern uint8 Loopback_bConfiguration;
extern uint8 Loopback_bDeviceAddress;
extern uint8 Loopback_bDeviceStatus;
extern uint8 Loopback_bEPHalt;
extern uint8 Loopback_bDevice;
extern uint8 Loopback_bTransferState;
extern T_Loopback_EP_CTL_BLOCK Loopback_EP[];

/*******************************************************************************
* Forward function references
********************************************************************************/
void Loopback_EnableInt(uint8 bVector, uint8 bMask);
void Loopback_SetVector(uint8 bVector, uint16 wAddress);
void Loopback_SetPriority(uint8 bVector, uint8 bPriority);
void Loopback_InitComponent(uint8 bDevice, uint8 bMode);

/*******************************************************************************
* External function references
********************************************************************************/
CY_ISR_PROTO(Loopback_EP_0_ISR);
CY_ISR_PROTO(Loopback_EP_1_ISR);
CY_ISR_PROTO(Loopback_EP_2_ISR);
CY_ISR_PROTO(Loopback_EP_3_ISR);
CY_ISR_PROTO(Loopback_EP_4_ISR);
CY_ISR_PROTO(Loopback_EP_5_ISR);
CY_ISR_PROTO(Loopback_EP_6_ISR);
CY_ISR_PROTO(Loopback_EP_7_ISR);
CY_ISR_PROTO(Loopback_EP_8_ISR);
CY_ISR_PROTO(Loopback_BUS_RESET_ISR);
CY_ISR_PROTO(Loopback_SOF_ISR);

/*******************************************************************************
* Function Name: Loopback_Start
********************************************************************************
* Summary:
*   This function initialize the USB SIE, arbiter and the
*   endpoint APIs, including setting the D+ Pullup
*
* Parameters:  
*   None
*******************************************************************************/
void Loopback_Start(uint8 bDevice, uint8 bMode)
{
    uint16 i;
    uint8 *p = &Loopback_ARB_RW1_DR[0];
    

   /* Set the 24 MHz Trim for USB */
    CY_SET_REG8((void Loopback_XDATA *)(CYDEV_MFGCFG_IMO_TR1), (CY_GET_XTND_REG8(CYDEV_FLSHID_BASE + 0x10F)));
    /* Enable USB clock */
    CY_SET_REG8(CYDEV_FASTCLK_IMO_CR, CY_GET_REG8(CYDEV_FASTCLK_IMO_CR) | Loopback_FASTCLK_IMO_CR_USBCLK_ON);
    /* Enable USB block */
    CY_SET_REG8(CYDEV_PM_ACT_CFG5, CY_GET_REG8(CYDEV_PM_ACT_CFG5) | 0x01);
    //TODO: pbvr: disable PM available for USB
    CY_SET_REG8(CYDEV_PM_AVAIL_CR6, CY_GET_REG8(CYDEV_PM_AVAIL_CR6) & ~0x10);
    /* Enable core clock */
    CY_SET_REG8(Loopback_USB_CLK_EN, 0x01);
    
    /* Bus Reset Length */
    CY_SET_REG8(Loopback_BUS_RST_CNT, 3);
    // TODO: pbvr: Enable PM available for USB
    CY_SET_REG8(CYDEV_PM_AVAIL_CR6, CY_GET_REG8(CYDEV_PM_AVAIL_CR6) | 0x10);
    /* If VBUS Monitoring is enable, setup the DR in the port reg */
#if (Loopback_MON_VBUS == 1)
    CY_SET_REG8(Loopback_VBUS_DR, CY_GET_REG8(Loopback_VBUS_DR) | Loopback_VBUS_MASK);
#endif

    /* Write WAx */
    CY_SET_REG8(&Loopback_ARB_RW1_WA[0],     0);
    CY_SET_REG8(&Loopback_ARB_RW1_WA_MSB[0], 0);
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < 512; i++)
    {
        CY_SET_REG8(p, 0xFF);
    }

    /* Set the bus reset Interrupt. */
    Loopback_SetVector(Loopback_BUS_RESET_VECT_NUM,   Loopback_BUS_RESET_ISR);
    Loopback_SetPriority(Loopback_BUS_RESET_VECT_NUM, Loopback_BUS_RESET_PRIOR);
    Loopback_EnableInt(Loopback_BUS_RESET_VECT_NUM,   Loopback_BUS_RESET_MASK);

    /* Set the SOF Interrupt. */
    Loopback_SetVector(Loopback_SOF_VECT_NUM,   Loopback_SOF_ISR);
    Loopback_SetPriority(Loopback_SOF_VECT_NUM, Loopback_SOF_PRIOR);
    Loopback_EnableInt(Loopback_SOF_VECT_NUM,   Loopback_SOF_MASK);

    /* Set the Control Endpoint Interrupt. */
    Loopback_SetVector(Loopback_EP_0_VECT_NUM,   Loopback_EP_0_ISR);
    Loopback_SetPriority(Loopback_EP_0_VECT_NUM, Loopback_EP_0_PRIOR);
    Loopback_EnableInt(Loopback_EP_0_VECT_NUM,   Loopback_EP_0_MASK);

    /* Set the Data Endpoint 1 Interrupt. */
    Loopback_SetVector(Loopback_EP_1_VECT_NUM,   Loopback_EP_1_ISR);
    Loopback_SetPriority(Loopback_EP_1_VECT_NUM, Loopback_EP_1_PRIOR);
    Loopback_EnableInt(Loopback_EP_1_VECT_NUM,   Loopback_EP_1_MASK);

    /* Set the Data Endpoint 2 Interrupt. */
    Loopback_SetVector(Loopback_EP_2_VECT_NUM,   Loopback_EP_2_ISR);
    Loopback_SetPriority(Loopback_EP_2_VECT_NUM, Loopback_EP_2_PRIOR);
    Loopback_EnableInt(Loopback_EP_2_VECT_NUM,   Loopback_EP_2_MASK);

    /* Set the Data Endpoint 3 Interrupt. */
    Loopback_SetVector(Loopback_EP_3_VECT_NUM,   Loopback_EP_3_ISR);
    Loopback_SetPriority(Loopback_EP_3_VECT_NUM, Loopback_EP_3_PRIOR);
    Loopback_EnableInt(Loopback_EP_3_VECT_NUM,   Loopback_EP_3_MASK);

    /* Set the Data Endpoint 4 Interrupt. */
    Loopback_SetVector(Loopback_EP_4_VECT_NUM,   Loopback_EP_4_ISR);
    Loopback_SetPriority(Loopback_EP_4_VECT_NUM, Loopback_EP_4_PRIOR);
    Loopback_EnableInt(Loopback_EP_4_VECT_NUM,   Loopback_EP_4_MASK);

    /* Set the Data Endpoint 5 Interrupt. */
    Loopback_SetVector(Loopback_EP_5_VECT_NUM,   Loopback_EP_5_ISR);
    Loopback_SetPriority(Loopback_EP_5_VECT_NUM, Loopback_EP_5_PRIOR);
    Loopback_EnableInt(Loopback_EP_5_VECT_NUM,   Loopback_EP_5_MASK);

    /* Set the Data Endpoint 6 Interrupt. */
    Loopback_SetVector(Loopback_EP_6_VECT_NUM,   Loopback_EP_6_ISR);
    Loopback_SetPriority(Loopback_EP_6_VECT_NUM, Loopback_EP_6_PRIOR);
    Loopback_EnableInt(Loopback_EP_6_VECT_NUM,   Loopback_EP_6_MASK);

     /* Set the Data Endpoint 7 Interrupt. */
    Loopback_SetVector(Loopback_EP_7_VECT_NUM,   Loopback_EP_7_ISR);
    Loopback_SetPriority(Loopback_EP_7_VECT_NUM, Loopback_EP_7_PRIOR);
    Loopback_EnableInt(Loopback_EP_7_VECT_NUM,   Loopback_EP_7_MASK);

    /* Set the Data Endpoint 8 Interrupt. */
    Loopback_SetVector(Loopback_EP_8_VECT_NUM,   Loopback_EP_8_ISR);
    Loopback_SetPriority(Loopback_EP_8_VECT_NUM, Loopback_EP_8_PRIOR);
    Loopback_EnableInt(Loopback_EP_8_VECT_NUM,   Loopback_EP_8_MASK);
    
    Loopback_InitComponent(bDevice, bMode);
}
/*******************************************************************************
* Function Name: Loopback_InitComponent
********************************************************************************
* Summary:
*   Initialize the component, except for the HW which is done one time in
*	the Start function.  This function pulls up D+.
*
* Parameters:  
*   None
*******************************************************************************/
void Loopback_InitComponent(uint8 bDevice, uint8 bMode)
{
    /* USB Locking: Enabled, VRegulator: Disabled */
    CY_SET_REG8(Loopback_CR1, (bMode | 0x02));

    /* Record the descriptor selection */
    Loopback_bDevice = bDevice;

    /* Clear all of the component data */
    Loopback_bConfiguration = 0;
    Loopback_bDeviceAddress  = 0;
    Loopback_bEPHalt = 0;
    Loopback_bDeviceStatus = 0;
    Loopback_bDeviceStatus = 0;
    Loopback_bTransferState = Loopback_TRANS_STATE_IDLE;

    /* STALL_IN_OUT */
    CY_SET_REG8(Loopback_EP0_CR, Loopback_MODE_STALL_IN_OUT);
    /* Enable the SIE with an address 0 */
    CY_SET_REG8(Loopback_CR0, Loopback_CR0_ENABLE );
    /* Finally, Enable d+ pullup */
    CY_SET_REG8(Loopback_USBIO_CR1, Loopback_USBIO_CR1_USBPUEN);
}
/*******************************************************************************
* Function Name: Loopback_Stop
********************************************************************************
* Summary:
*   This function shuts down the USB function including to release
*   the D+ Pullup and disabling the SIE.
*
* Parameters:  
*   None
*******************************************************************************/
void Loopback_Stop(void)
{
    /* Disable the SIE with address 0 */
    CY_SET_REG8(Loopback_CR0, 0x00);
    /* Disable the d+ pullup */
    CY_SET_REG8(Loopback_USBIO_CR1, 0x00);
    /* Disable the reset interrupt */
// TODO:	Loopback_DisableInt(Loopback_BUS_RESET_VECT_NUM,   Loopback_BUS_RESET_MASK);
// TODO:	Loopback_EnableInt(Loopback_EP_0_VECT_NUM,   Loopback_EP_0_MASK);

    Loopback_bConfiguration = 0;
    Loopback_bDeviceAddress  = 0;
    Loopback_bEPHalt = 0;
    Loopback_bDeviceStatus = 0;
}
/*******************************************************************************
* Function Name: Loopback_bCheckActivity
********************************************************************************
* Summary:
*   Returns the activity status of the bus.  Clears the status hardware to
*   provide fresh activity status on the next call of this routine.
*
* Parameters:  
*   None
*******************************************************************************/
uint8  Loopback_bCheckActivity(void)
{
    uint8 r = ((CY_GET_REG8(Loopback_CR1) >> 2) & Loopback_CR1_BUS_ACTIVITY);

    CY_SET_REG8(Loopback_CR1, (CY_GET_REG8(Loopback_CR1) & Loopback_CR1_BUS_ACTIVITY));

    return r;
}
/*******************************************************************************
* Function Name: Loopback_bGetConfiguration
********************************************************************************
* Summary:
*   Returns the current configuration setting
*
* Parameters:  
*   None
*******************************************************************************/
uint8  Loopback_bGetConfiguration(void)
{
    return Loopback_bConfiguration;
}
/*******************************************************************************
* Function Name: Loopback_bGetEPState
********************************************************************************
* Summary:
*   Returned the state of the requested endpoint.
*
* Parameters:  
*   bEPNumber: Endpoint Number
*******************************************************************************/
uint8 Loopback_bGetEPState(uint8 bEPNumber)
{
    return Loopback_EP[bEPNumber].bAPIEPState;
}
/*******************************************************************************
* Function Name: Loopback_wGetEPCount
********************************************************************************
* Summary:
*   Returns the transfer count for the requested endpoint.  The the value from
*   the count registers includes 2 counts for the two byte checksum of the
*   packet.  This function subtracts the two counts.
*
* Parameters:  
*   bEPNumber: Endpoint Number
*******************************************************************************/
uint16 Loopback_wGetEPCount(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);

    return ((CY_GET_REG8(&Loopback_SIE_EP1_CNT0[ri]) <<8) | CY_GET_REG8(&Loopback_SIE_EP1_CNT1[ri]) - 2);
}
/*******************************************************************************
* Function Name: Loopback_LoadEP
********************************************************************************
* Summary:  Load the endpoint buffer, set up the address pointers and go.
*
* Parameters:  
*   None
*******************************************************************************/
void   Loopback_LoadEP(uint8 bEPNumber, uint8 *pData, uint16 wLength)
{
    uint8 i;
    uint8 ri = ((bEPNumber - 1) << 4);
    uint8 *p = &Loopback_ARB_RW1_DR[ri];
    
    /* Write WAx */
    CY_SET_REG8(&Loopback_ARB_RW1_WA[ri],        Loopback_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&Loopback_ARB_RW1_WA_MSB[ri],    (Loopback_EP[bEPNumber].wBuffOffset >> 8));
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < wLength; i++)
    {
        CY_SET_REG8(p, *pData++);
    }
    /* Set the count and data toggle */
    CY_SET_REG8(&Loopback_SIE_EP1_CNT0[ri], (wLength >> 8) | (Loopback_EP[bEPNumber].bEPToggle));
    CY_SET_REG8(&Loopback_SIE_EP1_CNT1[ri],  wLength & 0xFFu);
    /* Write the RAx */
    CY_SET_REG8(&Loopback_ARB_RW1_RA[ri],        Loopback_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&Loopback_ARB_RW1_RA_MSB[ri],    (Loopback_EP[bEPNumber].wBuffOffset >> 8));

    /* Mark the event pending */
    Loopback_EP[bEPNumber].bAPIEPState = Loopback_EVENT_PENDING;
    /* Write the Mode register */
    CY_SET_REG8(&Loopback_SIE_EP1_CR0[ri], Loopback_EP[bEPNumber].bEPMode);
}

/*******************************************************************************
* Function Name: Loopback_bReadOutEP
********************************************************************************
* Summary:
*   Read data from an endpoint.  The application must call @INSTANCE_NAME`_bGetEPState
*   to see if an event is pending.
*
* Parameters:  
*   bEPNumber   Endpoint Number
*   pData       Pointer to the destination buffer
*   wLength     Length of the destination buffer
*
* Returns:
*   Number of bytes received
*
*******************************************************************************/
uint16 Loopback_ReadOutEP(uint8 bEPNumber, uint8 *pData, uint16 wLength)
{
    uint8 i;
    uint8 ri = ((bEPNumber - 1) << 4);
    uint8 *p = &Loopback_ARB_RW1_DR[ri];
    
    /* Write the RAx */
    CY_SET_REG8(&Loopback_ARB_RW1_RA[ri],        Loopback_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&Loopback_ARB_RW1_RA_MSB[ri],    (Loopback_EP[bEPNumber].wBuffOffset >> 8));
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < wLength; i++)
    {
        *pData++ = CY_GET_REG8(p);
    }
    /* Write the WAx */
    CY_SET_REG8(&Loopback_ARB_RW1_WA[ri],        Loopback_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&Loopback_ARB_RW1_WA_MSB[ri],    (Loopback_EP[bEPNumber].wBuffOffset >> 8));

    return wLength;
}
/*******************************************************************************
* Function Name: Loopback_EnableOutEP
********************************************************************************
* Summary:
*
* Parameters:  
*   None
*******************************************************************************/
void Loopback_EnableOutEP(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);
    /* Mark the event pending */
    Loopback_EP[bEPNumber].bAPIEPState = Loopback_NO_EVENT_PENDING;
    /* Write the Mode register */
    CY_SET_REG8(&Loopback_SIE_EP1_CR0[ri], Loopback_EP[bEPNumber].bEPMode);
}
/*******************************************************************************
* Function Name: Loopback_Force
********************************************************************************
* Summary:  Forces the bus state
*
* Parameters:  
*   bState 
*    Loopback_FORCE_J 
*    Loopback_FORCE_K 
*    Loopback_FORCE_SE0 
*    Loopback_FORCE_NONE
*
*******************************************************************************/
void Loopback_Force(uint8 bState)
{
    CY_SET_REG8(Loopback_USBIO_CR0, bState);
}
/*******************************************************************************
* Function Name: Loopback_bGetEPAckState
********************************************************************************
* Summary:  Returns the ACK of the CR0 Register (ACKD)
*
* Parameters:  
*   bEPNumber   Endpoint Number
*
* Returns
*   0 if nothing has been ACKD, non-=zero something has been ACKD
*******************************************************************************/
uint8 Loopback_bGetEPAckState(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);
  
    return (CY_GET_REG8(&Loopback_SIE_EP1_CR0[ri]) & Loopback_MODE_ACKD );
}
/*******************************************************************************
* Function Name: Loopback_SetPowerStatus
********************************************************************************
* Summary:
*   Sets the device power status for reporting in the Get Device Status
*   request
*
* Parameters:  
*   bPowerStaus 0 = Bus Powered, 1 = Self Powered
*
*******************************************************************************/
void Loopback_SetPowerStatus(uint8 bPowerStaus)
{
    if (bPowerStaus)
    {
        Loopback_bDeviceStatus |=  Loopback_DEVICE_STATUS_SELF_POWERED;
    }
    else
    {
        Loopback_bDeviceStatus &=  ~Loopback_DEVICE_STATUS_SELF_POWERED;
    }
}

/*******************************************************************************
* Function Name: Loopback_bVBusPresent
********************************************************************************
* Summary:
*   Determines VBUS presense for Self Powered Devices.  Returns 1 if VBUS
*   is present, otherwise 0.
*
* Parameters:  
*   None
*******************************************************************************/
#if (Loopback_MON_VBUS == 1)
uint8 Loopback_bVBusPresent()
{
    return ((CY_GET_REG8(Loopback_VBUS_PS) & Loopback_VBUS_MASK) ? 1 : 0);
}
#endif
/*******************************************************************************
* Function Name: Loopback_SetPriority
********************************************************************************
* Summary:  Set the priority for the endpoint interrupt
*
* Parameters:  
*   None
*******************************************************************************/
void Loopback_SetPriority(uint8 bVector, uint8 bPriority)
{
    CY_SET_REG8(&Loopback_USB_ISR_PRIOR[bVector], bPriority);
}

/*******************************************************************************
* Function Name: Loopback_SetVector
********************************************************************************
* Summary:  Set the address for the endpoint interrupt
*
* Parameters:  
*   None
*******************************************************************************/
void Loopback_SetVector(uint8 bVector, uint16 wAddress)
{
    CY_SET_REG16(&Loopback_USB_ISR_VECT[bVector], (uint16) wAddress);
}

/*******************************************************************************
* Function Name: Loopback_EnableInt
********************************************************************************
* Summary:  Enable an interrupt
*
* Parameters:  
*   None
*******************************************************************************/
void Loopback_EnableInt(uint8 bVector, uint8 bMask)
{
    CY_SET_REG8(&Loopback_USB_ISR_SET_EN[bVector / 8], bMask);
}

/*******************************************************************************
* Function Name: Loopback_DisableInt
********************************************************************************
* Summary:  Disable an interrupt
*
* Parameters:  
*   None
*******************************************************************************/
void Loopback_DisableInt(uint8 bVector, uint8 bMask)
{
    CY_SET_REG8(&Loopback_USB_ISR_CLR_EN[bVector / 8], bMask);
}
