/*******************************************************************************
* File Name: Loopback_drv.c
* Version 0.2
*
*  Description:
*    Endpoint 0 Driver for the USBFS Component.
*
*   Note:
*
********************************************************************************
* Copyright (c) 2008, 2009 Cypress Semiconductor Corporation.
********************************************************************************
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express, written permission of Cypress.
*******************************************************************************/

#include "cydevice.h"
#include "cyfitter.h"
#include "Loopback.h"

/*******************************************************************************
* Forward references for the EP0 ISR
********************************************************************************/
void  Loopback_HandleSetup(void);
void  Loopback_HandleIN(void);
void  Loopback_HandleOUT(void);
uint8 Loopback_InitControlRead(void);
void  Loopback_ControlReadDataStage(void);
void  Loopback_ControlReadStatusStage(void);
void  Loopback_ControlReadPrematureStatus(void);
uint8 Loopback_InitControlWrite(void);
void  Loopback_ControlWriteDataStage(void);
void  Loopback_ControlWriteStatusStage(void);
void  Loopback_ControlWritePrematureStatus(void);
uint8 Loopback_InitNoDataControlTransfer(void);
void  Loopback_NoDataControlStatusStage(void);
void Loopback_InitializeStatusBlock(void);
void Loopback_UpdateStatusBlock(uint8 bCompletionCode);

/*******************************************************************************
* Request Handlers
********************************************************************************/
uint8 Loopback_HandleStandardRqst(void);
uint8 Loopback_DispatchClassRqst(void);
uint8 Loopback_HandleVendorRqst(void);

/*******************************************************************************
* External data references
********************************************************************************/

/*******************************************************************************
* Global data allocation
********************************************************************************/
T_Loopback_EP_CTL_BLOCK Loopback_EP[9];
uint8 Loopback_bEPHalt;
uint8 Loopback_bConfiguration;
uint8 Loopback_bDeviceAddress;
uint8 Loopback_bDeviceStatus;
uint8 Loopback_bDevice;
/*******************************************************************************
* Local data allocation
********************************************************************************/
uint8 Loopback_bEP0Toggle;
uint8 Loopback_bLastPacketSize;
uint8 Loopback_bTransferState;
T_Loopback_TD CurrentTD;
uint8 Loopback_bEP0Mode;
uint8 Loopback_bEP0Count;
uint16 Loopback_TransferByteCount;

/*******************************************************************************
* Function Name: Loopback_ep_0_Interrupt
********************************************************************************
* Summary:
*   This Interrupt Service Routine handles Endpoint 0 (Control Pipe) traffic.  It
*   dispactches setup requests and handles the data and status stages.
*   
* Parameters:  
*   None
*******************************************************************************/
CY_ISR(Loopback_EP_0_ISR)
{
	uint8 bRegTemp = *Loopback_EP0_CR;
		
	if (!bRegTemp & Loopback_MODE_ACKD) return;             
	if (bRegTemp & Loopback_MODE_SETUP_RCVD) {
		Loopback_HandleSetup();
	} 
	else if (bRegTemp & Loopback_MODE_IN_RCVD) {
		Loopback_HandleIN();	
	}
	else if (bRegTemp & Loopback_MODE_OUT_RCVD) {
		Loopback_HandleOUT();
	}
	else {
//		ASSERT(0);
	}
	CY_SET_REG8(Loopback_EP0_CNT, Loopback_bEP0Toggle | Loopback_bEP0Count);
	/* Set the Mode Register  */
	CY_SET_REG8(Loopback_EP0_CR, Loopback_bEP0Mode);
}
/*******************************************************************************
* Function Name: Loopback_HandleSetup
********************************************************************************
* Summary:
*   This Routine dispatches requests for the four USB request types
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_HandleSetup(void)
{
	uint8 bRequestHandled;
    /* In case the previous transfer did not complete, close it out */
    Loopback_UpdateStatusBlock(Loopback_XFER_PREMATURE);
    
    switch (CY_GET_REG8(Loopback_bmRequestType) & Loopback_RQST_TYPE_MASK)
	{
	case Loopback_RQST_TYPE_STD:
		bRequestHandled = Loopback_HandleStandardRqst();
		break;
	case Loopback_RQST_TYPE_CLS:
		bRequestHandled = Loopback_DispatchClassRqst();
		break;
	case Loopback_RQST_TYPE_VND:
		bRequestHandled = Loopback_HandleVendorRqst();
		break;
	default:
		bRequestHandled = Loopback_FALSE;
		break;
	}
	if (bRequestHandled == Loopback_FALSE){
		Loopback_bEP0Mode = Loopback_MODE_STALL_IN_OUT;
	}
}
/*******************************************************************************
* Function Name: Loopback_HandleIN
********************************************************************************
* Summary:
*   This routine handles EP0 IN transfers.
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_HandleIN(void)
{
	switch (Loopback_bTransferState)
	{
	case Loopback_TRANS_STATE_IDLE:
		break;
	case Loopback_TRANS_STATE_CONTROL_READ:
		Loopback_ControlReadDataStage();
		break;
	case Loopback_TRANS_STATE_CONTROL_WRITE:
		Loopback_ControlReadStatusStage();
		break;
	case Loopback_TRANS_STATE_NO_DATA_CONTROL:
		Loopback_NoDataControlStatusStage();
		break;
	default:
		break;	
	}
}
/*******************************************************************************
* Function Name: Loopback_HandleOUT
********************************************************************************
* Summary:
*   This routine handles EP0 OUT transfers.
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_HandleOUT(void)
{
	switch (Loopback_bTransferState)
	{
	case Loopback_TRANS_STATE_IDLE:
		break;
	case Loopback_TRANS_STATE_CONTROL_READ:
		Loopback_ControlReadStatusStage();
		break;
	case Loopback_TRANS_STATE_CONTROL_WRITE:
		Loopback_ControlWriteDataStage();
		break;
	case Loopback_TRANS_STATE_NO_DATA_CONTROL:
        /* Update the completion block */
        Loopback_UpdateStatusBlock(Loopback_XFER_ERROR);
        /* We expect no more data, so stall INs and OUTs */
        Loopback_bEP0Mode = Loopback_MODE_STALL_IN_OUT;
		break;
	default:
		break;	
	}
}
/*******************************************************************************
* Function Name: Loopback_LoadEP0
********************************************************************************
* Summary:
*   This routine loads the EP0 data registers for OUT transfers.  It uses the
*   CurrentTD (previously initialized by the _InitControlWrite function and
*   updated for each OUT transfer, and the bLastPacketSize) to determine how
*   many uint8s to transfer on the current OUT.
*
*   If the number of uint8s remaining is zero and the last transfer was full, 
*   we need to send a zero length packet.  Otherwise we send the minimum
*   of the control endpoint size (8) or remaining number of uint8s for the
*   transaction.
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_LoadEP0(void)
{
	/* Update the transfer byte count from the last transaction */
    Loopback_TransferByteCount += Loopback_bLastPacketSize;
    /* Now load the next transaction */
    Loopback_bEP0Count = 0;
	while ((CurrentTD.wCount > 0) && (Loopback_bEP0Count < 8))
	{
        Loopback_EP0_DR0[Loopback_bEP0Count] = *CurrentTD.pData++;
		Loopback_bEP0Count++;
		CurrentTD.wCount--;
	}

	if ((Loopback_bEP0Count > 0) || (Loopback_bLastPacketSize == 8))
	{
		/* Update the data toggle */
		Loopback_bEP0Toggle ^= Loopback_EP0_CNT_DATA_TOGGLE;
		/* Set the Mode Register  */
		Loopback_bEP0Mode = Loopback_MODE_ACK_IN_STATUS_OUT;
		/* Update the state (or stay the same) */
		Loopback_bTransferState = Loopback_TRANS_STATE_CONTROL_READ;
	}
	else
	{
		/* Expect Status Stage Out */
		Loopback_bEP0Mode = Loopback_MODE_STATUS_OUT_ONLY;
		/* Update the state (or stay the same) */
		Loopback_bTransferState = Loopback_TRANS_STATE_CONTROL_READ;
	}

	/* Save the packet size for next time */
	Loopback_bLastPacketSize = Loopback_bEP0Count;
}
/*******************************************************************************
* Function Name: Loopback_InitControlRead
********************************************************************************
* Summary:
*   Initialize a control read transaction
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 Loopback_InitControlRead(void)
{
    uint16 wXferCount;

    /* Set up the state machine */
    Loopback_bTransferState = Loopback_TRANS_STATE_CONTROL_READ;
    /* Set the toggle, it gets updated in LoadEP */
    Loopback_bEP0Toggle = 0;
    /* Initialize the Status Block */
    Loopback_InitializeStatusBlock();
    wXferCount = ((*Loopback_wLengthHi << 8) | (*Loopback_wLengthLo));

    if (CurrentTD.wCount > wXferCount)
    {
        CurrentTD.wCount = wXferCount;
    }
    Loopback_LoadEP0();

    return Loopback_TRUE;
}
/*******************************************************************************
* Function Name: Loopback_ControlReadDataStage
********************************************************************************
* Summary:
*   Handle the Data Stage of a control read transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_ControlReadDataStage(void)
{
	Loopback_LoadEP0();
}
/*******************************************************************************
* Function Name: Loopback_ControlReadStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a control read transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_ControlReadStatusStage(void)
{
	/* Update the transfer byte count */
    Loopback_TransferByteCount += Loopback_bLastPacketSize;
    /* Go Idle */
    Loopback_bTransferState = Loopback_TRANS_STATE_IDLE;
    /* Update the completion block */
    Loopback_UpdateStatusBlock(Loopback_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	Loopback_bEP0Mode = Loopback_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: Loopback_InitControlWrite
********************************************************************************
* Summary:
*   Initialize a control write transaction
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 Loopback_InitControlWrite(void)
{
    uint16 wXferCount;
    
    /* Set up the state machine */
    Loopback_bTransferState = Loopback_TRANS_STATE_CONTROL_WRITE;
    /* This migh not be necessary */
    Loopback_bEP0Toggle = Loopback_EP0_CNT_DATA_TOGGLE;;
    /* Initialize the Status Block */
    Loopback_InitializeStatusBlock();

    wXferCount = ((CY_GET_REG8(Loopback_wLengthHi) << 8) | (CY_GET_REG8(Loopback_wLengthLo)));

    if (CurrentTD.wCount > wXferCount)
    {
        CurrentTD.wCount = wXferCount;
    }
    
	/* Expect Data or Status Stage */
	Loopback_bEP0Mode = Loopback_MODE_ACK_OUT_STATUS_IN;
    
    return Loopback_TRUE;
}
/*******************************************************************************
* Function Name: Loopback_ControlWriteDataStage
********************************************************************************
* Summary:
*   Handle the Data Stage of a control write transfer
*       1. Get the data (We assume the destination was validated previously)
*       3. Update the count and data toggle
*       2. Update the mode register for the next transaction
* Parameters:  
*   None
*******************************************************************************/
void Loopback_ControlWriteDataStage(void)
{
	uint8 *pReg = Loopback_EP0_DR0; 
	Loopback_bEP0Count = (CY_GET_REG8(Loopback_EP0_CNT ) & 0x0F) - 2;
    
    Loopback_TransferByteCount += Loopback_bEP0Count;
    
    while ((CurrentTD.wCount > 0) && (Loopback_bEP0Count > 0))
	{
        *CurrentTD.pData++ = CY_GET_REG8(pReg++);
        Loopback_bEP0Count--;
		CurrentTD.wCount--;
	}
	/* Update the data toggle */
	Loopback_bEP0Toggle ^= Loopback_EP0_CNT_DATA_TOGGLE;
	/* Expect Data or Status Stage */
	Loopback_bEP0Mode = Loopback_MODE_ACK_OUT_STATUS_IN;
}
/*******************************************************************************
* Function Name: Loopback_ControlWriteStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a control write transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_ControlWriteStatusStage(void)
{
	/* Go Idle */
    Loopback_bTransferState = Loopback_TRANS_STATE_IDLE;
    /* Update the completion block */
    Loopback_UpdateStatusBlock(Loopback_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	Loopback_bEP0Mode = Loopback_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: Loopback_InitNoDataControlTransfer
********************************************************************************
* Summary:
*   Initialize a no data control transfer
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 Loopback_InitNoDataControlTransfer(void)
{
	Loopback_bTransferState = Loopback_TRANS_STATE_NO_DATA_CONTROL;
	Loopback_bEP0Mode = Loopback_MODE_STATUS_IN_ONLY;
	return Loopback_TRUE;
}
/*******************************************************************************
* Function Name: Loopback_NoDataControlStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a no data control transfer.
*
*   SET_ADDRESS is special, since we need to receive the status stage with
*   the old address.
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_NoDataControlStatusStage(void)
{
	/* Change the USB address register if we got a SET_ADDRESS. */
    if (Loopback_bDeviceAddress != 0)
    {
        CY_SET_REG8(Loopback_CR0, Loopback_bDeviceAddress | Loopback_CR0_ENABLE);
        Loopback_bDeviceAddress = 0;
    }
    	/* Go Idle */
    Loopback_bTransferState = Loopback_TRANS_STATE_IDLE;
    /* Update the completion block */
    Loopback_UpdateStatusBlock(Loopback_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	Loopback_bEP0Mode = Loopback_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: Loopback_UpdateStatusBlock
********************************************************************************
* Summary:
*   Update the Completion Status Block for a Request.  The block is updated
*   with the completion code the Loopback_TransferByteCount.  The
*   StatusBlock Pointer  is set to NULL.
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_UpdateStatusBlock(uint8 bCompletionCode)
{
    if (CurrentTD.pStatusBlock != Loopback_NULL)
    {
        CurrentTD.pStatusBlock->bStatus = bCompletionCode;
        CurrentTD.pStatusBlock->wLength = Loopback_TransferByteCount;
        CurrentTD.pStatusBlock = Loopback_NULL;
    }
}
/*******************************************************************************
* Function Name: Loopback_InitializeStatusBlock
********************************************************************************
* Summary:
*   Initialize the Completion Status Block for a Request.  The completion
*   code is set to USB_XFER_IDLE.
*
*   Also, initializes Loopback_TransferByteCount.  Save some space,
*   this is the only consumer.
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_InitializeStatusBlock(void)
{
    Loopback_TransferByteCount = 0;
    if (CurrentTD.pStatusBlock != Loopback_NULL)
    {
        CurrentTD.pStatusBlock->bStatus = Loopback_XFER_IDLE;
        CurrentTD.pStatusBlock->wLength = 0;
    }
}