/*******************************************************************************
* File Name: Loopback_std.c
* Version 0.2
*
*  Description:
*    USB Standard request handler.
*
*   Note:
*
********************************************************************************
* Copyright (2008), Cypress Semiconductor Corporation.
********************************************************************************
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express, written permission of Cypress.
*******************************************************************************/

#include "cydevice.h"
#include "cyfitter.h"
#include "Loopback.h"

/*******************************************************************************
* External references
********************************************************************************/
uint8 Loopback_InitControlRead(void);
uint8 Loopback_InitControlWrite(void);
uint8 Loopback_InitNoDataControlTransfer(void);
uint8 Loopback_DispatchClassRqst(void);
extern uint8 Loopback_CODE Loopback_DEVICE0_DESCR[];
extern uint8 Loopback_CODE Loopback_DEVICE0_CONFIGURATION0_DESCR[]; 
extern uint8 Loopback_CODE Loopback_STRING_DESCRIPTORS[];

extern uint8 Loopback_bEPHalt;
extern uint8 Loopback_bDevice;
extern uint8 Loopback_bConfiguration;
extern uint8 Loopback_bDeviceAddress;
extern uint8 Loopback_bDeviceStatus;
extern T_Loopback_LUT Loopback_CODE Loopback_TABLE[];
extern T_Loopback_EP_CTL_BLOCK Loopback_EP[];
extern T_Loopback_TD CurrentTD;
uint8 tBuffer[2];

/*******************************************************************************
* Forward references
********************************************************************************/
void Loopback_Config(void);
T_Loopback_LUT *Loopback_GetConfigTablePtr(uint8 c);
T_Loopback_LUT *Loopback_GetDeviceTablePtr();
uint8 Loopback_ClearEndpointHalt(void);
uint8 Loopback_SetEndpointHalt(void);
//uint8 Loopback_GetDeviceStatus(void);

/*******************************************************************************
* Function Name: Loopback_HandleStandardRqst
********************************************************************************
* Summary:
*   This Routine dispatches standard requests
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 Loopback_HandleStandardRqst()
{
    uint8 bRequestHandled = Loopback_FALSE;
    uint8 *pStr;
    uint8 nStr;
    uint16 wCount;
    
    T_Loopback_LUT *pTmp;
    CurrentTD.wCount = 0;
    
    if ((CY_GET_REG8(Loopback_bmRequestType) & Loopback_RQST_DIR_MASK) == Loopback_RQST_DIR_D2H)
    {
        /* Control Read */
        switch (CY_GET_REG8(Loopback_bRequest)) 
        {
        case Loopback_GET_DESCRIPTOR:
            if (CY_GET_REG8(Loopback_wValueHi) == Loopback_DESCR_DEVICE)  
            {
                pTmp = Loopback_GetDeviceTablePtr();
                CurrentTD.pData     = pTmp->p_list;
                CurrentTD.wCount    = 18;
                bRequestHandled     = Loopback_InitControlRead();
                }
            else if (CY_GET_REG8(Loopback_wValueHi) == Loopback_DESCR_CONFIG)  {
                pTmp = Loopback_GetConfigTablePtr(CY_GET_REG8(Loopback_wValueLo));
                CurrentTD.pData     = pTmp->p_list;
                wCount    = ((CurrentTD.pData)[3] <<8) | (CurrentTD.pData)[2];
                CurrentTD.wCount    = wCount;
                bRequestHandled     = Loopback_InitControlRead();
            }
#if defined(Loopback_ENABLE_STRINGS)
            else if (CY_GET_REG8(Loopback_wValueHi) == Loopback_DESCR_STRING)
            {
#if defined(EE_STRING)

#endif
                pStr = &Loopback_STRING_DESCRIPTORS[0];
                nStr = 0;
                while ((CY_GET_REG8(Loopback_wValueLo) > nStr) && (*pStr != 0)){
                    pStr += *pStr;
                    nStr++;
                };

                if (*pStr != 0)
                {
                    CurrentTD.wCount    = *pStr;
                    CurrentTD.pData     = pStr;
                    bRequestHandled     = Loopback_InitControlRead();
                }
            }
#endif
            else
            {
                bRequestHandled = Loopback_DispatchClassRqst();
            }
            break;
        case Loopback_GET_STATUS:
            switch ((CY_GET_REG8(Loopback_bmRequestType) & Loopback_RQST_RCPT_MASK))
            {
            case Loopback_RQST_RCPT_EP:
                    CurrentTD.wCount    = 1;
                    CurrentTD.pData     = &Loopback_EP[CY_GET_REG8(Loopback_wIndexLo) & 0x07].bHWEPState;
                    bRequestHandled     = Loopback_InitControlRead();
                break;
            case Loopback_RQST_RCPT_DEV:
                CurrentTD.wCount    = 2;
                tBuffer[0] = Loopback_bDeviceStatus;
                tBuffer[1] = 0;
                CurrentTD.pData     = &tBuffer[0];
                bRequestHandled     = Loopback_InitControlRead();
                break;
            }
            break;
        case Loopback_GET_CONFIGURATION:
            CurrentTD.wCount    = 1;
            CurrentTD.pData     = &Loopback_bConfiguration;
            bRequestHandled     = Loopback_InitControlRead();
            break;
        default:
            break;
        }
    }
    else {
        /* Control Write */
        switch (CY_GET_REG8(Loopback_bRequest)) 
        {
        case Loopback_SET_ADDRESS:
            Loopback_bDeviceAddress = (CY_GET_REG8(Loopback_wValueLo));
            bRequestHandled = Loopback_InitNoDataControlTransfer();
            break;
        case Loopback_SET_CONFIGURATION:
            Loopback_bConfiguration	= CY_GET_REG8(Loopback_wValueLo);
            Loopback_Config();
            bRequestHandled	= Loopback_InitNoDataControlTransfer();
            break;
        case Loopback_CLEAR_FEATURE:
            switch ((CY_GET_REG8(Loopback_bmRequestType) & Loopback_RQST_RCPT_MASK))
            {
            case Loopback_RQST_RCPT_EP:
                if (CY_GET_REG8(Loopback_wValueLo) ==Loopback_ENDPOINT_HALT)
                {
                    bRequestHandled	= Loopback_ClearEndpointHalt();
                }
                break;
            case Loopback_RQST_RCPT_DEV:
                /* Clear device REMOTE_WAKEUP */
                Loopback_bDeviceStatus &= ~Loopback_DEVICE_STATUS_REMOTE_WAKEUP;
                bRequestHandled	= Loopback_InitNoDataControlTransfer();
                break;
            }
            break;
		case Loopback_SET_FEATURE:
            switch ((CY_GET_REG8(Loopback_bmRequestType) & Loopback_RQST_RCPT_MASK))
            {
            case Loopback_RQST_RCPT_EP:
                if (CY_GET_REG8(Loopback_wValueLo) ==Loopback_ENDPOINT_HALT)
                {
                    bRequestHandled	= Loopback_SetEndpointHalt();
                }
                break;
            case Loopback_RQST_RCPT_DEV:
                /* Clear device REMOTE_WAKEUP */
                Loopback_bDeviceStatus &= ~Loopback_DEVICE_STATUS_REMOTE_WAKEUP;
                bRequestHandled	= Loopback_InitNoDataControlTransfer();
                break;
            }
        default:
            break;
        }
    
    }

    return bRequestHandled;
}    
/*******************************************************************************
* Function Name: Loopback_Config
********************************************************************************
* Summary:
*   This routine configures endpoints for the entire configuration by scanning
*   the configuration descriptor.  It configures the bAlternateSetting 0 for
*   each interface.
*   
* Parameters:  
*   None
*******************************************************************************/
void Loopback_Config()
{
    uint8 ep,cur_ep,i;
    uint8 iso, bCount;
    T_Loopback_LUT *pTmp;
    T_Loopback_EP_SETTINGS_BLOCK *pEP;

    /* Clear all of the endpoints */
    for (ep = 0; ep < 9; ep++)
    {
        Loopback_EP[ep].bAttrib = 0;
        Loopback_EP[ep].bHWEPState = 0;
        Loopback_EP[ep].bAPIEPState = Loopback_NO_EVENT_PENDING;;
        Loopback_EP[ep].bEPToggle = 0;
        Loopback_EP[ep].bEPMode = 0;
        Loopback_EP[ep].wBufferSize = 0;
    }

    pTmp = Loopback_GetConfigTablePtr(Loopback_bConfiguration - 1);
    pTmp++;
    ep = pTmp->c;                                       /* For this table, c is the number of endpoints  */
    pEP = (T_Loopback_EP_SETTINGS_BLOCK *) pTmp->p_list;     /* and p_list points the endpoint setting table. */
    
    for (i = 0; i < ep; i++, pEP++)
    {
        cur_ep = pEP->bAddr; // & 0x03;
        cur_ep &= 0x03;
        iso  = ((pEP->bmAttributes & Loopback_EP_TYPE_MASK) == Loopback_EP_TYPE_ISOC);
        if (pEP->bAddr & Loopback_DIR_IN)
        {
        /* IN Endpoint */
            Loopback_EP[cur_ep].bAPIEPState = Loopback_EVENT_PENDING;
            Loopback_EP[cur_ep].bEPMode = (iso ? Loopback_MODE_ISO_IN : Loopback_MODE_ACK_IN);
        }
        else
        {
        /* OUT Endpoint */
            Loopback_EP[cur_ep].bAPIEPState = Loopback_NO_EVENT_PENDING;
            Loopback_EP[cur_ep].bEPMode = (iso ? Loopback_MODE_ISO_OUT : Loopback_MODE_ACK_OUT);
        }
        Loopback_EP[cur_ep].wBufferSize = pEP->wBufferSize;
        Loopback_EP[cur_ep].bAddr = pEP->bAddr;
    } 

    /* Set the endpoint buffer addresses */
    bCount = 0;
    ep = 1;
    for (i = 0; i < 0x80; i+= 0x10)
    {
        CY_SET_REG8(&Loopback_ARB_EP1_CFG[i], Loopback_ARB_EPX_CFG_CRC_BYPASS);
        
        
        CY_SET_REG8(&Loopback_SIE_EP1_CR0[i], Loopback_MODE_NAK_IN_OUT);

        Loopback_EP[ep].wBuffOffset = bCount;
        bCount += Loopback_EP[ep].wBufferSize;
        CY_SET_REG8(&Loopback_SIE_EP1_CNT0[i],  (Loopback_EP[ep].wBufferSize >> 8));
        CY_SET_REG8(&Loopback_SIE_EP1_CNT1[i],  Loopback_EP[ep].wBufferSize & 0xFFu);

        CY_SET_REG8(&Loopback_ARB_RW1_RA[i],        Loopback_EP[ep].wBuffOffset & 0xFFu);
        CY_SET_REG8(&Loopback_ARB_RW1_RA_MSB[i],    (Loopback_EP[ep].wBuffOffset >> 8));
        CY_SET_REG8(&Loopback_ARB_RW1_WA[i],        Loopback_EP[ep].wBuffOffset & 0xFFu);
        CY_SET_REG8(&Loopback_ARB_RW1_WA_MSB[i],    (Loopback_EP[ep].wBuffOffset >> 8));
        
        ep++;
    }
    CY_SET_REG8(&Loopback_SIE_EP_INT_EN, 0xFF);
}
/*******************************************************************************
* Function Name: Loopback_GetConfigTablePtr
********************************************************************************
* Summary:
*   This routine returns a pointer a configuration table entry
*   
* Parameters:  
*   c Configuration Index
*******************************************************************************/
T_Loopback_LUT *Loopback_GetConfigTablePtr(uint8 c)
{
    uint8 cfg;
    /* Device Table */
    T_Loopback_LUT *pTmp = Loopback_GetDeviceTablePtr();

    /* The first entry points to the Device Descriptor.  */
    pTmp++;
    /* Top of the Configuration Table List*/
    pTmp = pTmp->p_list;

    /* Now get the config */
    for (cfg = c; cfg > 0; cfg--)
    {
        pTmp++;
    }
    return pTmp;
}
/*******************************************************************************
* Function Name: Loopback_GetDeviceTablePtr
********************************************************************************
* Summary:
*   This routine returns a pointer to the current config table based on the
*   selector in the request wValueLo
*   
* Parameters:  
*   None
*******************************************************************************/
T_Loopback_LUT *Loopback_GetDeviceTablePtr()
{
    /* Device Table */
    return Loopback_TABLE[Loopback_bDevice].p_list;
}

/*******************************************************************************
* Function Name: Loopback_SetEndpointHalt
********************************************************************************
* Summary:
*   This routine handles set endpoint halt
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 Loopback_SetEndpointHalt()
{
    uint8 ep, ri;
    uint8 bRequestHandled = Loopback_FALSE;

    /* Clear endpoint halt */
    ep = (CY_GET_REG8(Loopback_wIndexLo) & 0x07);
    ri = ((ep - 1) << 4);
    if (ep != 0)
    {
        /* Set the endpoint Halt */
        Loopback_EP[ep].bHWEPState |= (Loopback_ENDPOINT_STATUS_HALT);
        
        /* Clear the data toggle */
        Loopback_EP[ep].bEPToggle = 0;
        
        if (Loopback_EP[ep].bAddr & Loopback_DIR_IN)
        {
            /* IN Endpoint */
            Loopback_EP[ep].bAPIEPState = Loopback_NO_EVENT_ALLOWED;
            CY_SET_REG8(&Loopback_SIE_EP1_CR0[ri], (Loopback_MODE_STALL_DATA_EP | Loopback_MODE_ACK_IN));
        }
        else
        {
            /* OUT Endpoint */
            Loopback_EP[ep].bAPIEPState = Loopback_NO_EVENT_ALLOWED;
            CY_SET_REG8(&Loopback_SIE_EP1_CR0[ri], (Loopback_MODE_STALL_DATA_EP | Loopback_MODE_ACK_OUT));
        }
        bRequestHandled = Loopback_InitNoDataControlTransfer();
    }
    return bRequestHandled;
}
/*******************************************************************************
* Function Name: Loopback_ClearEndpointHalt
********************************************************************************
* Summary:
*   This routine handles clear endpoint halt
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 Loopback_ClearEndpointHalt()
{
    uint8 ep, ri;
    uint8 bRequestHandled = Loopback_FALSE;

    /* Clear endpoint halt */
    ep = (CY_GET_REG8(Loopback_wIndexLo) & 0x07);
    ri = ((ep  - 1) << 4);

    if (ep != 0)
    {
        /* Set the endpoint Halt */
        Loopback_EP[ep].bHWEPState &= ~(Loopback_ENDPOINT_STATUS_HALT);
        
        /* Clear the data toggle */
        Loopback_EP[ep].bEPToggle = 0;
        
        if (Loopback_EP[ep].bAddr & Loopback_DIR_IN)
        {
            /* IN Endpoint */
            Loopback_EP[ep].bAPIEPState = Loopback_NO_EVENT_PENDING;
            CY_SET_REG8(&Loopback_SIE_EP1_CR0[ri], Loopback_MODE_NAK_IN);
        }
        else
        {
            /* OUT Endpoint */
            Loopback_EP[ep].bAPIEPState = Loopback_EVENT_PENDING;
            CY_SET_REG8(&Loopback_SIE_EP1_CR0[ri], Loopback_MODE_ACK_OUT);
        }
        bRequestHandled = Loopback_InitNoDataControlTransfer();
    }
    return bRequestHandled;
}