/*******************************************************************************
* File Name: USBFS_1.c
* Version 0.2
*
*  Description:
*    API for USBFS Component.
*
*   Note:
*   
*   Many of the functions use endpoint number.  RAM arrays are sized with 9
*   elements so they are indexed directly by bEPNumber.  The SIE and ARB
*   registers are indexed by variations of bEPNumber - 1.
*
********************************************************************************
* Copyright (c) 2009, Cypress Semiconductor Corporation.
********************************************************************************
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express, written permission of Cypress.
*******************************************************************************/

#include "device.h"
#include "cyfitter.h"
#include "USBFS_1.h"

/*******************************************************************************
* External data references
********************************************************************************/
extern uint8 USBFS_1_bConfiguration;
extern uint8 USBFS_1_bDeviceAddress;
extern uint8 USBFS_1_bDeviceStatus;
extern uint8 USBFS_1_bEPHalt;
extern uint8 USBFS_1_bDevice;
extern uint8 USBFS_1_bTransferState;
extern T_USBFS_1_EP_CTL_BLOCK USBFS_1_EP[];

/*******************************************************************************
* Forward function references
********************************************************************************/
void USBFS_1_EnableInt(uint8 bVector, uint8 bMask);
void USBFS_1_SetVector(uint8 bVector, uint16 wAddress);
void USBFS_1_SetPriority(uint8 bVector, uint8 bPriority);
void USBFS_1_InitComponent(uint8 bDevice, uint8 bMode);

/*******************************************************************************
* External function references
********************************************************************************/
CY_ISR_PROTO(USBFS_1_EP_0_ISR);
CY_ISR_PROTO(USBFS_1_EP_1_ISR);
CY_ISR_PROTO(USBFS_1_EP_2_ISR);
CY_ISR_PROTO(USBFS_1_EP_3_ISR);
CY_ISR_PROTO(USBFS_1_EP_4_ISR);
CY_ISR_PROTO(USBFS_1_EP_5_ISR);
CY_ISR_PROTO(USBFS_1_EP_6_ISR);
CY_ISR_PROTO(USBFS_1_EP_7_ISR);
CY_ISR_PROTO(USBFS_1_EP_8_ISR);
CY_ISR_PROTO(USBFS_1_BUS_RESET_ISR);
CY_ISR_PROTO(USBFS_1_SOF_ISR);

/*******************************************************************************
* Function Name: USBFS_1_Start
********************************************************************************
* Summary:
*   This function initialize the USB SIE, arbiter and the
*   endpoint APIs, including setting the D+ Pullup
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_Start(uint8 bDevice, uint8 bMode)
{
    uint16 i;
    uint8 *p = &USBFS_1_ARB_RW1_DR[0];
    

   /* Set the 24 MHz Trim for USB */
    CY_SET_REG8((void USBFS_1_XDATA *)(CYDEV_MFGCFG_IMO_TR1), (CY_GET_XTND_REG8(CYDEV_FLSHID_BASE + 0x10F)));
    /* Enable USB clock */
    CY_SET_REG8(CYDEV_FASTCLK_IMO_CR, CY_GET_REG8(CYDEV_FASTCLK_IMO_CR) | USBFS_1_FASTCLK_IMO_CR_USBCLK_ON);
    /* Enable USB block */
    CY_SET_REG8(CYDEV_PM_ACT_CFG5, CY_GET_REG8(CYDEV_PM_ACT_CFG5) | 0x01);
    //TODO: pbvr: disable PM available for USB
    CY_SET_REG8(CYDEV_PM_AVAIL_CR6, CY_GET_REG8(CYDEV_PM_AVAIL_CR6) & ~0x10);
    /* Enable core clock */
    CY_SET_REG8(USBFS_1_USB_CLK_EN, 0x01);
    
    /* Bus Reset Length */
    CY_SET_REG8(USBFS_1_BUS_RST_CNT, 3);
    // TODO: pbvr: Enable PM available for USB
    CY_SET_REG8(CYDEV_PM_AVAIL_CR6, CY_GET_REG8(CYDEV_PM_AVAIL_CR6) | 0x10);
    /* If VBUS Monitoring is enable, setup the DR in the port reg */
#if (USBFS_1_MON_VBUS == 1)
    CY_SET_REG8(USBFS_1_VBUS_DR, CY_GET_REG8(USBFS_1_VBUS_DR) | USBFS_1_VBUS_MASK);
#endif

    /* Write WAx */
    CY_SET_REG8(&USBFS_1_ARB_RW1_WA[0],     0);
    CY_SET_REG8(&USBFS_1_ARB_RW1_WA_MSB[0], 0);
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < 512; i++)
    {
        CY_SET_REG8(p, 0xFF);
    }

    /* Set the bus reset Interrupt. */
    USBFS_1_SetVector(USBFS_1_BUS_RESET_VECT_NUM,   USBFS_1_BUS_RESET_ISR);
    USBFS_1_SetPriority(USBFS_1_BUS_RESET_VECT_NUM, USBFS_1_BUS_RESET_PRIOR);
    USBFS_1_EnableInt(USBFS_1_BUS_RESET_VECT_NUM,   USBFS_1_BUS_RESET_MASK);

    /* Set the SOF Interrupt. */
    USBFS_1_SetVector(USBFS_1_SOF_VECT_NUM,   USBFS_1_SOF_ISR);
    USBFS_1_SetPriority(USBFS_1_SOF_VECT_NUM, USBFS_1_SOF_PRIOR);
    USBFS_1_EnableInt(USBFS_1_SOF_VECT_NUM,   USBFS_1_SOF_MASK);

    /* Set the Control Endpoint Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_0_VECT_NUM,   USBFS_1_EP_0_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_0_VECT_NUM, USBFS_1_EP_0_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_0_VECT_NUM,   USBFS_1_EP_0_MASK);

    /* Set the Data Endpoint 1 Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_1_VECT_NUM,   USBFS_1_EP_1_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_1_VECT_NUM, USBFS_1_EP_1_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_1_VECT_NUM,   USBFS_1_EP_1_MASK);

    /* Set the Data Endpoint 2 Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_2_VECT_NUM,   USBFS_1_EP_2_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_2_VECT_NUM, USBFS_1_EP_2_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_2_VECT_NUM,   USBFS_1_EP_2_MASK);

    /* Set the Data Endpoint 3 Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_3_VECT_NUM,   USBFS_1_EP_3_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_3_VECT_NUM, USBFS_1_EP_3_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_3_VECT_NUM,   USBFS_1_EP_3_MASK);

    /* Set the Data Endpoint 4 Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_4_VECT_NUM,   USBFS_1_EP_4_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_4_VECT_NUM, USBFS_1_EP_4_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_4_VECT_NUM,   USBFS_1_EP_4_MASK);

    /* Set the Data Endpoint 5 Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_5_VECT_NUM,   USBFS_1_EP_5_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_5_VECT_NUM, USBFS_1_EP_5_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_5_VECT_NUM,   USBFS_1_EP_5_MASK);

    /* Set the Data Endpoint 6 Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_6_VECT_NUM,   USBFS_1_EP_6_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_6_VECT_NUM, USBFS_1_EP_6_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_6_VECT_NUM,   USBFS_1_EP_6_MASK);

     /* Set the Data Endpoint 7 Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_7_VECT_NUM,   USBFS_1_EP_7_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_7_VECT_NUM, USBFS_1_EP_7_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_7_VECT_NUM,   USBFS_1_EP_7_MASK);

    /* Set the Data Endpoint 8 Interrupt. */
    USBFS_1_SetVector(USBFS_1_EP_8_VECT_NUM,   USBFS_1_EP_8_ISR);
    USBFS_1_SetPriority(USBFS_1_EP_8_VECT_NUM, USBFS_1_EP_8_PRIOR);
    USBFS_1_EnableInt(USBFS_1_EP_8_VECT_NUM,   USBFS_1_EP_8_MASK);
    
    USBFS_1_InitComponent(bDevice, bMode);
}
/*******************************************************************************
* Function Name: USBFS_1_InitComponent
********************************************************************************
* Summary:
*   Initialize the component, except for the HW which is done one time in
*	the Start function.  This function pulls up D+.
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_InitComponent(uint8 bDevice, uint8 bMode)
{
    /* USB Locking: Enabled, VRegulator: Disabled */
    CY_SET_REG8(USBFS_1_CR1, (bMode | 0x02));

    /* Record the descriptor selection */
    USBFS_1_bDevice = bDevice;

    /* Clear all of the component data */
    USBFS_1_bConfiguration = 0;
    USBFS_1_bDeviceAddress  = 0;
    USBFS_1_bEPHalt = 0;
    USBFS_1_bDeviceStatus = 0;
    USBFS_1_bDeviceStatus = 0;
    USBFS_1_bTransferState = USBFS_1_TRANS_STATE_IDLE;

    /* STALL_IN_OUT */
    CY_SET_REG8(USBFS_1_EP0_CR, USBFS_1_MODE_STALL_IN_OUT);
    /* Enable the SIE with an address 0 */
    CY_SET_REG8(USBFS_1_CR0, USBFS_1_CR0_ENABLE );
    /* Finally, Enable d+ pullup */
    CY_SET_REG8(USBFS_1_USBIO_CR1, USBFS_1_USBIO_CR1_USBPUEN);
}
/*******************************************************************************
* Function Name: USBFS_1_Stop
********************************************************************************
* Summary:
*   This function shuts down the USB function including to release
*   the D+ Pullup and disabling the SIE.
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_Stop(void)
{
    /* Disable the SIE with address 0 */
    CY_SET_REG8(USBFS_1_CR0, 0x00);
    /* Disable the d+ pullup */
    CY_SET_REG8(USBFS_1_USBIO_CR1, 0x00);
    /* Disable the reset interrupt */
// TODO:	USBFS_1_DisableInt(USBFS_1_BUS_RESET_VECT_NUM,   USBFS_1_BUS_RESET_MASK);
// TODO:	USBFS_1_EnableInt(USBFS_1_EP_0_VECT_NUM,   USBFS_1_EP_0_MASK);

    USBFS_1_bConfiguration = 0;
    USBFS_1_bDeviceAddress  = 0;
    USBFS_1_bEPHalt = 0;
    USBFS_1_bDeviceStatus = 0;
}
/*******************************************************************************
* Function Name: USBFS_1_bCheckActivity
********************************************************************************
* Summary:
*   Returns the activity status of the bus.  Clears the status hardware to
*   provide fresh activity status on the next call of this routine.
*
* Parameters:  
*   None
*******************************************************************************/
uint8  USBFS_1_bCheckActivity(void)
{
    uint8 r = ((CY_GET_REG8(USBFS_1_CR1) >> 2) & USBFS_1_CR1_BUS_ACTIVITY);

    CY_SET_REG8(USBFS_1_CR1, (CY_GET_REG8(USBFS_1_CR1) & USBFS_1_CR1_BUS_ACTIVITY));

    return r;
}
/*******************************************************************************
* Function Name: USBFS_1_bGetConfiguration
********************************************************************************
* Summary:
*   Returns the current configuration setting
*
* Parameters:  
*   None
*******************************************************************************/
uint8  USBFS_1_bGetConfiguration(void)
{
    return USBFS_1_bConfiguration;
}
/*******************************************************************************
* Function Name: USBFS_1_bGetEPState
********************************************************************************
* Summary:
*   Returned the state of the requested endpoint.
*
* Parameters:  
*   bEPNumber: Endpoint Number
*******************************************************************************/
uint8 USBFS_1_bGetEPState(uint8 bEPNumber)
{
    return USBFS_1_EP[bEPNumber].bAPIEPState;
}
/*******************************************************************************
* Function Name: USBFS_1_wGetEPCount
********************************************************************************
* Summary:
*   Returns the transfer count for the requested endpoint.  The the value from
*   the count registers includes 2 counts for the two byte checksum of the
*   packet.  This function subtracts the two counts.
*
* Parameters:  
*   bEPNumber: Endpoint Number
*******************************************************************************/
uint16 USBFS_1_wGetEPCount(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);

    return ((CY_GET_REG8(&USBFS_1_SIE_EP1_CNT0[ri]) <<8) | CY_GET_REG8(&USBFS_1_SIE_EP1_CNT1[ri]) - 2);
}
/*******************************************************************************
* Function Name: USBFS_1_LoadEP
********************************************************************************
* Summary:  Load the endpoint buffer, set up the address pointers and go.
*
* Parameters:  
*   None
*******************************************************************************/
void   USBFS_1_LoadEP(uint8 bEPNumber, uint8 *pData, uint16 wLength)
{
    uint8 i;
    uint8 ri = ((bEPNumber - 1) << 4);
    uint8 *p = &USBFS_1_ARB_RW1_DR[ri];
    
    /* Write WAx */
    CY_SET_REG8(&USBFS_1_ARB_RW1_WA[ri],        USBFS_1_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&USBFS_1_ARB_RW1_WA_MSB[ri],    (USBFS_1_EP[bEPNumber].wBuffOffset >> 8));
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < wLength; i++)
    {
        CY_SET_REG8(p, *pData++);
    }
    /* Set the count and data toggle */
    CY_SET_REG8(&USBFS_1_SIE_EP1_CNT0[ri], (wLength >> 8) | (USBFS_1_EP[bEPNumber].bEPToggle));
    CY_SET_REG8(&USBFS_1_SIE_EP1_CNT1[ri],  wLength & 0xFFu);
    /* Write the RAx */
    CY_SET_REG8(&USBFS_1_ARB_RW1_RA[ri],        USBFS_1_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&USBFS_1_ARB_RW1_RA_MSB[ri],    (USBFS_1_EP[bEPNumber].wBuffOffset >> 8));

    /* Mark the event pending */
    USBFS_1_EP[bEPNumber].bAPIEPState = USBFS_1_EVENT_PENDING;
    /* Write the Mode register */
    CY_SET_REG8(&USBFS_1_SIE_EP1_CR0[ri], USBFS_1_EP[bEPNumber].bEPMode);
}

/*******************************************************************************
* Function Name: USBFS_1_bReadOutEP
********************************************************************************
* Summary:
*   Read data from an endpoint.  The application must call @INSTANCE_NAME`_bGetEPState
*   to see if an event is pending.
*
* Parameters:  
*   bEPNumber   Endpoint Number
*   pData       Pointer to the destination buffer
*   wLength     Length of the destination buffer
*
* Returns:
*   Number of bytes received
*
*******************************************************************************/
uint16 USBFS_1_ReadOutEP(uint8 bEPNumber, uint8 *pData, uint16 wLength)
{
    uint8 i;
    uint8 ri = ((bEPNumber - 1) << 4);
    uint8 *p = &USBFS_1_ARB_RW1_DR[ri];
    
    /* Write the RAx */
    CY_SET_REG8(&USBFS_1_ARB_RW1_RA[ri],        USBFS_1_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&USBFS_1_ARB_RW1_RA_MSB[ri],    (USBFS_1_EP[bEPNumber].wBuffOffset >> 8));
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < wLength; i++)
    {
        *pData++ = CY_GET_REG8(p);
    }
    /* Write the WAx */
    CY_SET_REG8(&USBFS_1_ARB_RW1_WA[ri],        USBFS_1_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&USBFS_1_ARB_RW1_WA_MSB[ri],    (USBFS_1_EP[bEPNumber].wBuffOffset >> 8));

    return wLength;
}
/*******************************************************************************
* Function Name: USBFS_1_EnableOutEP
********************************************************************************
* Summary:
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_EnableOutEP(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);
    /* Mark the event pending */
    USBFS_1_EP[bEPNumber].bAPIEPState = USBFS_1_NO_EVENT_PENDING;
    /* Write the Mode register */
    CY_SET_REG8(&USBFS_1_SIE_EP1_CR0[ri], USBFS_1_EP[bEPNumber].bEPMode);
}
/*******************************************************************************
* Function Name: USBFS_1_Force
********************************************************************************
* Summary:  Forces the bus state
*
* Parameters:  
*   bState 
*    USBFS_1_FORCE_J 
*    USBFS_1_FORCE_K 
*    USBFS_1_FORCE_SE0 
*    USBFS_1_FORCE_NONE
*
*******************************************************************************/
void USBFS_1_Force(uint8 bState)
{
    CY_SET_REG8(USBFS_1_USBIO_CR0, bState);
}
/*******************************************************************************
* Function Name: USBFS_1_bGetEPAckState
********************************************************************************
* Summary:  Returns the ACK of the CR0 Register (ACKD)
*
* Parameters:  
*   bEPNumber   Endpoint Number
*
* Returns
*   0 if nothing has been ACKD, non-=zero something has been ACKD
*******************************************************************************/
uint8 USBFS_1_bGetEPAckState(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);
  
    return (CY_GET_REG8(&USBFS_1_SIE_EP1_CR0[ri]) & USBFS_1_MODE_ACKD );
}
/*******************************************************************************
* Function Name: USBFS_1_SetPowerStatus
********************************************************************************
* Summary:
*   Sets the device power status for reporting in the Get Device Status
*   request
*
* Parameters:  
*   bPowerStaus 0 = Bus Powered, 1 = Self Powered
*
*******************************************************************************/
void USBFS_1_SetPowerStatus(uint8 bPowerStaus)
{
    if (bPowerStaus)
    {
        USBFS_1_bDeviceStatus |=  USBFS_1_DEVICE_STATUS_SELF_POWERED;
    }
    else
    {
        USBFS_1_bDeviceStatus &=  ~USBFS_1_DEVICE_STATUS_SELF_POWERED;
    }
}

/*******************************************************************************
* Function Name: USBFS_1_bVBusPresent
********************************************************************************
* Summary:
*   Determines VBUS presense for Self Powered Devices.  Returns 1 if VBUS
*   is present, otherwise 0.
*
* Parameters:  
*   None
*******************************************************************************/
#if (USBFS_1_MON_VBUS == 1)
uint8 USBFS_1_bVBusPresent()
{
    return ((CY_GET_REG8(USBFS_1_VBUS_PS) & USBFS_1_VBUS_MASK) ? 1 : 0);
}
#endif
/*******************************************************************************
* Function Name: USBFS_1_SetPriority
********************************************************************************
* Summary:  Set the priority for the endpoint interrupt
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_SetPriority(uint8 bVector, uint8 bPriority)
{
    CY_SET_REG8(&USBFS_1_USB_ISR_PRIOR[bVector], bPriority);
}

/*******************************************************************************
* Function Name: USBFS_1_SetVector
********************************************************************************
* Summary:  Set the address for the endpoint interrupt
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_SetVector(uint8 bVector, uint16 wAddress)
{
    CY_SET_REG16(&USBFS_1_USB_ISR_VECT[bVector], (uint16) wAddress);
}

/*******************************************************************************
* Function Name: USBFS_1_EnableInt
********************************************************************************
* Summary:  Enable an interrupt
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_EnableInt(uint8 bVector, uint8 bMask)
{
    CY_SET_REG8(&USBFS_1_USB_ISR_SET_EN[bVector / 8], bMask);
}

/*******************************************************************************
* Function Name: USBFS_1_DisableInt
********************************************************************************
* Summary:  Disable an interrupt
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_DisableInt(uint8 bVector, uint8 bMask)
{
    CY_SET_REG8(&USBFS_1_USB_ISR_CLR_EN[bVector / 8], bMask);
}
