/*******************************************************************************
* File Name: USBFS_1_drv.c
* Version 0.2
*
*  Description:
*    Endpoint 0 Driver for the USBFS Component.
*
*   Note:
*
********************************************************************************
* Copyright (c) 2008, 2009 Cypress Semiconductor Corporation.
********************************************************************************
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express, written permission of Cypress.
*******************************************************************************/

#include "cydevice.h"
#include "cyfitter.h"
#include "USBFS_1.h"

/*******************************************************************************
* Forward references for the EP0 ISR
********************************************************************************/
void  USBFS_1_HandleSetup(void);
void  USBFS_1_HandleIN(void);
void  USBFS_1_HandleOUT(void);
uint8 USBFS_1_InitControlRead(void);
void  USBFS_1_ControlReadDataStage(void);
void  USBFS_1_ControlReadStatusStage(void);
void  USBFS_1_ControlReadPrematureStatus(void);
uint8 USBFS_1_InitControlWrite(void);
void  USBFS_1_ControlWriteDataStage(void);
void  USBFS_1_ControlWriteStatusStage(void);
void  USBFS_1_ControlWritePrematureStatus(void);
uint8 USBFS_1_InitNoDataControlTransfer(void);
void  USBFS_1_NoDataControlStatusStage(void);
void USBFS_1_InitializeStatusBlock(void);
void USBFS_1_UpdateStatusBlock(uint8 bCompletionCode);

/*******************************************************************************
* Request Handlers
********************************************************************************/
uint8 USBFS_1_HandleStandardRqst(void);
uint8 USBFS_1_DispatchClassRqst(void);
uint8 USBFS_1_HandleVendorRqst(void);

/*******************************************************************************
* External data references
********************************************************************************/

/*******************************************************************************
* Global data allocation
********************************************************************************/
T_USBFS_1_EP_CTL_BLOCK USBFS_1_EP[9];
uint8 USBFS_1_bEPHalt;
uint8 USBFS_1_bConfiguration;
uint8 USBFS_1_bDeviceAddress;
uint8 USBFS_1_bDeviceStatus;
uint8 USBFS_1_bDevice;
/*******************************************************************************
* Local data allocation
********************************************************************************/
uint8 USBFS_1_bEP0Toggle;
uint8 USBFS_1_bLastPacketSize;
uint8 USBFS_1_bTransferState;
T_USBFS_1_TD CurrentTD;
uint8 USBFS_1_bEP0Mode;
uint8 USBFS_1_bEP0Count;
uint16 USBFS_1_TransferByteCount;

/*******************************************************************************
* Function Name: USBFS_1_ep_0_Interrupt
********************************************************************************
* Summary:
*   This Interrupt Service Routine handles Endpoint 0 (Control Pipe) traffic.  It
*   dispactches setup requests and handles the data and status stages.
*   
* Parameters:  
*   None
*******************************************************************************/
CY_ISR(USBFS_1_EP_0_ISR)
{
	uint8 bRegTemp = *USBFS_1_EP0_CR;
		
	if (!bRegTemp & USBFS_1_MODE_ACKD) return;             
	if (bRegTemp & USBFS_1_MODE_SETUP_RCVD) {
		USBFS_1_HandleSetup();
	} 
	else if (bRegTemp & USBFS_1_MODE_IN_RCVD) {
		USBFS_1_HandleIN();	
	}
	else if (bRegTemp & USBFS_1_MODE_OUT_RCVD) {
		USBFS_1_HandleOUT();
	}
	else {
//		ASSERT(0);
	}
	CY_SET_REG8(USBFS_1_EP0_CNT, USBFS_1_bEP0Toggle | USBFS_1_bEP0Count);
	/* Set the Mode Register  */
	CY_SET_REG8(USBFS_1_EP0_CR, USBFS_1_bEP0Mode);
}
/*******************************************************************************
* Function Name: USBFS_1_HandleSetup
********************************************************************************
* Summary:
*   This Routine dispatches requests for the four USB request types
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_HandleSetup(void)
{
	uint8 bRequestHandled;
    /* In case the previous transfer did not complete, close it out */
    USBFS_1_UpdateStatusBlock(USBFS_1_XFER_PREMATURE);
    
    switch (CY_GET_REG8(USBFS_1_bmRequestType) & USBFS_1_RQST_TYPE_MASK)
	{
	case USBFS_1_RQST_TYPE_STD:
		bRequestHandled = USBFS_1_HandleStandardRqst();
		break;
	case USBFS_1_RQST_TYPE_CLS:
		bRequestHandled = USBFS_1_DispatchClassRqst();
		break;
	case USBFS_1_RQST_TYPE_VND:
		bRequestHandled = USBFS_1_HandleVendorRqst();
		break;
	default:
		bRequestHandled = USBFS_1_FALSE;
		break;
	}
	if (bRequestHandled == USBFS_1_FALSE){
		USBFS_1_bEP0Mode = USBFS_1_MODE_STALL_IN_OUT;
	}
}
/*******************************************************************************
* Function Name: USBFS_1_HandleIN
********************************************************************************
* Summary:
*   This routine handles EP0 IN transfers.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_HandleIN(void)
{
	switch (USBFS_1_bTransferState)
	{
	case USBFS_1_TRANS_STATE_IDLE:
		break;
	case USBFS_1_TRANS_STATE_CONTROL_READ:
		USBFS_1_ControlReadDataStage();
		break;
	case USBFS_1_TRANS_STATE_CONTROL_WRITE:
		USBFS_1_ControlReadStatusStage();
		break;
	case USBFS_1_TRANS_STATE_NO_DATA_CONTROL:
		USBFS_1_NoDataControlStatusStage();
		break;
	default:
		break;	
	}
}
/*******************************************************************************
* Function Name: USBFS_1_HandleOUT
********************************************************************************
* Summary:
*   This routine handles EP0 OUT transfers.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_HandleOUT(void)
{
	switch (USBFS_1_bTransferState)
	{
	case USBFS_1_TRANS_STATE_IDLE:
		break;
	case USBFS_1_TRANS_STATE_CONTROL_READ:
		USBFS_1_ControlReadStatusStage();
		break;
	case USBFS_1_TRANS_STATE_CONTROL_WRITE:
		USBFS_1_ControlWriteDataStage();
		break;
	case USBFS_1_TRANS_STATE_NO_DATA_CONTROL:
        /* Update the completion block */
        USBFS_1_UpdateStatusBlock(USBFS_1_XFER_ERROR);
        /* We expect no more data, so stall INs and OUTs */
        USBFS_1_bEP0Mode = USBFS_1_MODE_STALL_IN_OUT;
		break;
	default:
		break;	
	}
}
/*******************************************************************************
* Function Name: USBFS_1_LoadEP0
********************************************************************************
* Summary:
*   This routine loads the EP0 data registers for OUT transfers.  It uses the
*   CurrentTD (previously initialized by the _InitControlWrite function and
*   updated for each OUT transfer, and the bLastPacketSize) to determine how
*   many uint8s to transfer on the current OUT.
*
*   If the number of uint8s remaining is zero and the last transfer was full, 
*   we need to send a zero length packet.  Otherwise we send the minimum
*   of the control endpoint size (8) or remaining number of uint8s for the
*   transaction.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_LoadEP0(void)
{
	/* Update the transfer byte count from the last transaction */
    USBFS_1_TransferByteCount += USBFS_1_bLastPacketSize;
    /* Now load the next transaction */
    USBFS_1_bEP0Count = 0;
	while ((CurrentTD.wCount > 0) && (USBFS_1_bEP0Count < 8))
	{
        USBFS_1_EP0_DR0[USBFS_1_bEP0Count] = *CurrentTD.pData++;
		USBFS_1_bEP0Count++;
		CurrentTD.wCount--;
	}

	if ((USBFS_1_bEP0Count > 0) || (USBFS_1_bLastPacketSize == 8))
	{
		/* Update the data toggle */
		USBFS_1_bEP0Toggle ^= USBFS_1_EP0_CNT_DATA_TOGGLE;
		/* Set the Mode Register  */
		USBFS_1_bEP0Mode = USBFS_1_MODE_ACK_IN_STATUS_OUT;
		/* Update the state (or stay the same) */
		USBFS_1_bTransferState = USBFS_1_TRANS_STATE_CONTROL_READ;
	}
	else
	{
		/* Expect Status Stage Out */
		USBFS_1_bEP0Mode = USBFS_1_MODE_STATUS_OUT_ONLY;
		/* Update the state (or stay the same) */
		USBFS_1_bTransferState = USBFS_1_TRANS_STATE_CONTROL_READ;
	}

	/* Save the packet size for next time */
	USBFS_1_bLastPacketSize = USBFS_1_bEP0Count;
}
/*******************************************************************************
* Function Name: USBFS_1_InitControlRead
********************************************************************************
* Summary:
*   Initialize a control read transaction
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_1_InitControlRead(void)
{
    uint16 wXferCount;

    /* Set up the state machine */
    USBFS_1_bTransferState = USBFS_1_TRANS_STATE_CONTROL_READ;
    /* Set the toggle, it gets updated in LoadEP */
    USBFS_1_bEP0Toggle = 0;
    /* Initialize the Status Block */
    USBFS_1_InitializeStatusBlock();
    wXferCount = ((*USBFS_1_wLengthHi << 8) | (*USBFS_1_wLengthLo));

    if (CurrentTD.wCount > wXferCount)
    {
        CurrentTD.wCount = wXferCount;
    }
    USBFS_1_LoadEP0();

    return USBFS_1_TRUE;
}
/*******************************************************************************
* Function Name: USBFS_1_ControlReadDataStage
********************************************************************************
* Summary:
*   Handle the Data Stage of a control read transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_ControlReadDataStage(void)
{
	USBFS_1_LoadEP0();
}
/*******************************************************************************
* Function Name: USBFS_1_ControlReadStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a control read transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_ControlReadStatusStage(void)
{
	/* Update the transfer byte count */
    USBFS_1_TransferByteCount += USBFS_1_bLastPacketSize;
    /* Go Idle */
    USBFS_1_bTransferState = USBFS_1_TRANS_STATE_IDLE;
    /* Update the completion block */
    USBFS_1_UpdateStatusBlock(USBFS_1_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	USBFS_1_bEP0Mode = USBFS_1_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: USBFS_1_InitControlWrite
********************************************************************************
* Summary:
*   Initialize a control write transaction
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_1_InitControlWrite(void)
{
    uint16 wXferCount;
    
    /* Set up the state machine */
    USBFS_1_bTransferState = USBFS_1_TRANS_STATE_CONTROL_WRITE;
    /* This migh not be necessary */
    USBFS_1_bEP0Toggle = USBFS_1_EP0_CNT_DATA_TOGGLE;;
    /* Initialize the Status Block */
    USBFS_1_InitializeStatusBlock();

    wXferCount = ((CY_GET_REG8(USBFS_1_wLengthHi) << 8) | (CY_GET_REG8(USBFS_1_wLengthLo)));

    if (CurrentTD.wCount > wXferCount)
    {
        CurrentTD.wCount = wXferCount;
    }
    
	/* Expect Data or Status Stage */
	USBFS_1_bEP0Mode = USBFS_1_MODE_ACK_OUT_STATUS_IN;
    
    return USBFS_1_TRUE;
}
/*******************************************************************************
* Function Name: USBFS_1_ControlWriteDataStage
********************************************************************************
* Summary:
*   Handle the Data Stage of a control write transfer
*       1. Get the data (We assume the destination was validated previously)
*       3. Update the count and data toggle
*       2. Update the mode register for the next transaction
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_ControlWriteDataStage(void)
{
	uint8 *pReg = USBFS_1_EP0_DR0; 
	USBFS_1_bEP0Count = (CY_GET_REG8(USBFS_1_EP0_CNT ) & 0x0F) - 2;
    
    USBFS_1_TransferByteCount += USBFS_1_bEP0Count;
    
    while ((CurrentTD.wCount > 0) && (USBFS_1_bEP0Count > 0))
	{
        *CurrentTD.pData++ = CY_GET_REG8(pReg++);
        USBFS_1_bEP0Count--;
		CurrentTD.wCount--;
	}
	/* Update the data toggle */
	USBFS_1_bEP0Toggle ^= USBFS_1_EP0_CNT_DATA_TOGGLE;
	/* Expect Data or Status Stage */
	USBFS_1_bEP0Mode = USBFS_1_MODE_ACK_OUT_STATUS_IN;
}
/*******************************************************************************
* Function Name: USBFS_1_ControlWriteStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a control write transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_ControlWriteStatusStage(void)
{
	/* Go Idle */
    USBFS_1_bTransferState = USBFS_1_TRANS_STATE_IDLE;
    /* Update the completion block */
    USBFS_1_UpdateStatusBlock(USBFS_1_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	USBFS_1_bEP0Mode = USBFS_1_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: USBFS_1_InitNoDataControlTransfer
********************************************************************************
* Summary:
*   Initialize a no data control transfer
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_1_InitNoDataControlTransfer(void)
{
	USBFS_1_bTransferState = USBFS_1_TRANS_STATE_NO_DATA_CONTROL;
	USBFS_1_bEP0Mode = USBFS_1_MODE_STATUS_IN_ONLY;
	return USBFS_1_TRUE;
}
/*******************************************************************************
* Function Name: USBFS_1_NoDataControlStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a no data control transfer.
*
*   SET_ADDRESS is special, since we need to receive the status stage with
*   the old address.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_NoDataControlStatusStage(void)
{
	/* Change the USB address register if we got a SET_ADDRESS. */
    if (USBFS_1_bDeviceAddress != 0)
    {
        CY_SET_REG8(USBFS_1_CR0, USBFS_1_bDeviceAddress | USBFS_1_CR0_ENABLE);
        USBFS_1_bDeviceAddress = 0;
    }
    	/* Go Idle */
    USBFS_1_bTransferState = USBFS_1_TRANS_STATE_IDLE;
    /* Update the completion block */
    USBFS_1_UpdateStatusBlock(USBFS_1_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	USBFS_1_bEP0Mode = USBFS_1_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: USBFS_1_UpdateStatusBlock
********************************************************************************
* Summary:
*   Update the Completion Status Block for a Request.  The block is updated
*   with the completion code the USBFS_1_TransferByteCount.  The
*   StatusBlock Pointer  is set to NULL.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_UpdateStatusBlock(uint8 bCompletionCode)
{
    if (CurrentTD.pStatusBlock != USBFS_1_NULL)
    {
        CurrentTD.pStatusBlock->bStatus = bCompletionCode;
        CurrentTD.pStatusBlock->wLength = USBFS_1_TransferByteCount;
        CurrentTD.pStatusBlock = USBFS_1_NULL;
    }
}
/*******************************************************************************
* Function Name: USBFS_1_InitializeStatusBlock
********************************************************************************
* Summary:
*   Initialize the Completion Status Block for a Request.  The completion
*   code is set to USB_XFER_IDLE.
*
*   Also, initializes USBFS_1_TransferByteCount.  Save some space,
*   this is the only consumer.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_InitializeStatusBlock(void)
{
    USBFS_1_TransferByteCount = 0;
    if (CurrentTD.pStatusBlock != USBFS_1_NULL)
    {
        CurrentTD.pStatusBlock->bStatus = USBFS_1_XFER_IDLE;
        CurrentTD.pStatusBlock->wLength = 0;
    }
}