/*******************************************************************************
* File Name: USBFS_1_std.c
* Version 0.2
*
*  Description:
*    USB Standard request handler.
*
*   Note:
*
********************************************************************************
* Copyright (2008), Cypress Semiconductor Corporation.
********************************************************************************
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express, written permission of Cypress.
*******************************************************************************/

#include "cydevice.h"
#include "cyfitter.h"
#include "USBFS_1.h"

/*******************************************************************************
* External references
********************************************************************************/
uint8 USBFS_1_InitControlRead(void);
uint8 USBFS_1_InitControlWrite(void);
uint8 USBFS_1_InitNoDataControlTransfer(void);
uint8 USBFS_1_DispatchClassRqst(void);
extern uint8 USBFS_1_CODE USBFS_1_DEVICE0_DESCR[];
extern uint8 USBFS_1_CODE USBFS_1_DEVICE0_CONFIGURATION0_DESCR[]; 
extern uint8 USBFS_1_CODE USBFS_1_STRING_DESCRIPTORS[];

extern uint8 USBFS_1_bEPHalt;
extern uint8 USBFS_1_bDevice;
extern uint8 USBFS_1_bConfiguration;
extern uint8 USBFS_1_bDeviceAddress;
extern uint8 USBFS_1_bDeviceStatus;
extern T_USBFS_1_LUT USBFS_1_CODE USBFS_1_TABLE[];
extern T_USBFS_1_EP_CTL_BLOCK USBFS_1_EP[];
extern T_USBFS_1_TD CurrentTD;
uint8 tBuffer[2];

/*******************************************************************************
* Forward references
********************************************************************************/
void USBFS_1_Config(void);
T_USBFS_1_LUT *USBFS_1_GetConfigTablePtr(uint8 c);
T_USBFS_1_LUT *USBFS_1_GetDeviceTablePtr();
uint8 USBFS_1_ClearEndpointHalt(void);
uint8 USBFS_1_SetEndpointHalt(void);
//uint8 USBFS_1_GetDeviceStatus(void);

/*******************************************************************************
* Function Name: USBFS_1_HandleStandardRqst
********************************************************************************
* Summary:
*   This Routine dispatches standard requests
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_1_HandleStandardRqst()
{
    uint8 bRequestHandled = USBFS_1_FALSE;
    uint8 *pStr;
    uint8 nStr;
    uint16 wCount;
    
    T_USBFS_1_LUT *pTmp;
    CurrentTD.wCount = 0;
    
    if ((CY_GET_REG8(USBFS_1_bmRequestType) & USBFS_1_RQST_DIR_MASK) == USBFS_1_RQST_DIR_D2H)
    {
        /* Control Read */
        switch (CY_GET_REG8(USBFS_1_bRequest)) 
        {
        case USBFS_1_GET_DESCRIPTOR:
            if (CY_GET_REG8(USBFS_1_wValueHi) == USBFS_1_DESCR_DEVICE)  
            {
                pTmp = USBFS_1_GetDeviceTablePtr();
                CurrentTD.pData     = pTmp->p_list;
                CurrentTD.wCount    = 18;
                bRequestHandled     = USBFS_1_InitControlRead();
                }
            else if (CY_GET_REG8(USBFS_1_wValueHi) == USBFS_1_DESCR_CONFIG)  {
                pTmp = USBFS_1_GetConfigTablePtr(CY_GET_REG8(USBFS_1_wValueLo));
                CurrentTD.pData     = pTmp->p_list;
                wCount    = ((CurrentTD.pData)[3] <<8) | (CurrentTD.pData)[2];
                CurrentTD.wCount    = wCount;
                bRequestHandled     = USBFS_1_InitControlRead();
            }
#if defined(USBFS_1_ENABLE_STRINGS)
            else if (CY_GET_REG8(USBFS_1_wValueHi) == USBFS_1_DESCR_STRING)
            {
#if defined(EE_STRING)

#endif
                pStr = &USBFS_1_STRING_DESCRIPTORS[0];
                nStr = 0;
                while ((CY_GET_REG8(USBFS_1_wValueLo) > nStr) && (*pStr != 0)){
                    pStr += *pStr;
                    nStr++;
                };

                if (*pStr != 0)
                {
                    CurrentTD.wCount    = *pStr;
                    CurrentTD.pData     = pStr;
                    bRequestHandled     = USBFS_1_InitControlRead();
                }
            }
#endif
            else
            {
                bRequestHandled = USBFS_1_DispatchClassRqst();
            }
            break;
        case USBFS_1_GET_STATUS:
            switch ((CY_GET_REG8(USBFS_1_bmRequestType) & USBFS_1_RQST_RCPT_MASK))
            {
            case USBFS_1_RQST_RCPT_EP:
                    CurrentTD.wCount    = 1;
                    CurrentTD.pData     = &USBFS_1_EP[CY_GET_REG8(USBFS_1_wIndexLo) & 0x07].bHWEPState;
                    bRequestHandled     = USBFS_1_InitControlRead();
                break;
            case USBFS_1_RQST_RCPT_DEV:
                CurrentTD.wCount    = 2;
                tBuffer[0] = USBFS_1_bDeviceStatus;
                tBuffer[1] = 0;
                CurrentTD.pData     = &tBuffer[0];
                bRequestHandled     = USBFS_1_InitControlRead();
                break;
            }
            break;
        case USBFS_1_GET_CONFIGURATION:
            CurrentTD.wCount    = 1;
            CurrentTD.pData     = &USBFS_1_bConfiguration;
            bRequestHandled     = USBFS_1_InitControlRead();
            break;
        default:
            break;
        }
    }
    else {
        /* Control Write */
        switch (CY_GET_REG8(USBFS_1_bRequest)) 
        {
        case USBFS_1_SET_ADDRESS:
            USBFS_1_bDeviceAddress = (CY_GET_REG8(USBFS_1_wValueLo));
            bRequestHandled = USBFS_1_InitNoDataControlTransfer();
            break;
        case USBFS_1_SET_CONFIGURATION:
            USBFS_1_bConfiguration	= CY_GET_REG8(USBFS_1_wValueLo);
            USBFS_1_Config();
            bRequestHandled	= USBFS_1_InitNoDataControlTransfer();
            break;
        case USBFS_1_CLEAR_FEATURE:
            switch ((CY_GET_REG8(USBFS_1_bmRequestType) & USBFS_1_RQST_RCPT_MASK))
            {
            case USBFS_1_RQST_RCPT_EP:
                if (CY_GET_REG8(USBFS_1_wValueLo) ==USBFS_1_ENDPOINT_HALT)
                {
                    bRequestHandled	= USBFS_1_ClearEndpointHalt();
                }
                break;
            case USBFS_1_RQST_RCPT_DEV:
                /* Clear device REMOTE_WAKEUP */
                USBFS_1_bDeviceStatus &= ~USBFS_1_DEVICE_STATUS_REMOTE_WAKEUP;
                bRequestHandled	= USBFS_1_InitNoDataControlTransfer();
                break;
            }
            break;
		case USBFS_1_SET_FEATURE:
            switch ((CY_GET_REG8(USBFS_1_bmRequestType) & USBFS_1_RQST_RCPT_MASK))
            {
            case USBFS_1_RQST_RCPT_EP:
                if (CY_GET_REG8(USBFS_1_wValueLo) ==USBFS_1_ENDPOINT_HALT)
                {
                    bRequestHandled	= USBFS_1_SetEndpointHalt();
                }
                break;
            case USBFS_1_RQST_RCPT_DEV:
                /* Clear device REMOTE_WAKEUP */
                USBFS_1_bDeviceStatus &= ~USBFS_1_DEVICE_STATUS_REMOTE_WAKEUP;
                bRequestHandled	= USBFS_1_InitNoDataControlTransfer();
                break;
            }
        default:
            break;
        }
    
    }

    return bRequestHandled;
}    
/*******************************************************************************
* Function Name: USBFS_1_Config
********************************************************************************
* Summary:
*   This routine configures endpoints for the entire configuration by scanning
*   the configuration descriptor.  It configures the bAlternateSetting 0 for
*   each interface.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_1_Config()
{
    uint8 ep,cur_ep,i;
    uint8 iso, bCount;
    T_USBFS_1_LUT *pTmp;
    T_USBFS_1_EP_SETTINGS_BLOCK *pEP;

    /* Clear all of the endpoints */
    for (ep = 0; ep < 9; ep++)
    {
        USBFS_1_EP[ep].bAttrib = 0;
        USBFS_1_EP[ep].bHWEPState = 0;
        USBFS_1_EP[ep].bAPIEPState = USBFS_1_NO_EVENT_PENDING;;
        USBFS_1_EP[ep].bEPToggle = 0;
        USBFS_1_EP[ep].bEPMode = 0;
        USBFS_1_EP[ep].wBufferSize = 0;
    }

    pTmp = USBFS_1_GetConfigTablePtr(USBFS_1_bConfiguration - 1);
    pTmp++;
    ep = pTmp->c;                                       /* For this table, c is the number of endpoints  */
    pEP = (T_USBFS_1_EP_SETTINGS_BLOCK *) pTmp->p_list;     /* and p_list points the endpoint setting table. */
    
    for (i = 0; i < ep; i++, pEP++)
    {
        cur_ep = pEP->bAddr; // & 0x03;
        cur_ep &= 0x03;
        iso  = ((pEP->bmAttributes & USBFS_1_EP_TYPE_MASK) == USBFS_1_EP_TYPE_ISOC);
        if (pEP->bAddr & USBFS_1_DIR_IN)
        {
        /* IN Endpoint */
            USBFS_1_EP[cur_ep].bAPIEPState = USBFS_1_EVENT_PENDING;
            USBFS_1_EP[cur_ep].bEPMode = (iso ? USBFS_1_MODE_ISO_IN : USBFS_1_MODE_ACK_IN);
        }
        else
        {
        /* OUT Endpoint */
            USBFS_1_EP[cur_ep].bAPIEPState = USBFS_1_NO_EVENT_PENDING;
            USBFS_1_EP[cur_ep].bEPMode = (iso ? USBFS_1_MODE_ISO_OUT : USBFS_1_MODE_ACK_OUT);
        }
        USBFS_1_EP[cur_ep].wBufferSize = pEP->wBufferSize;
        USBFS_1_EP[cur_ep].bAddr = pEP->bAddr;
    } 

    /* Set the endpoint buffer addresses */
    bCount = 0;
    ep = 1;
    for (i = 0; i < 0x80; i+= 0x10)
    {
        CY_SET_REG8(&USBFS_1_ARB_EP1_CFG[i], USBFS_1_ARB_EPX_CFG_CRC_BYPASS);
        
        
        CY_SET_REG8(&USBFS_1_SIE_EP1_CR0[i], USBFS_1_MODE_NAK_IN_OUT);

        USBFS_1_EP[ep].wBuffOffset = bCount;
        bCount += USBFS_1_EP[ep].wBufferSize;
        CY_SET_REG8(&USBFS_1_SIE_EP1_CNT0[i],  (USBFS_1_EP[ep].wBufferSize >> 8));
        CY_SET_REG8(&USBFS_1_SIE_EP1_CNT1[i],  USBFS_1_EP[ep].wBufferSize & 0xFFu);

        CY_SET_REG8(&USBFS_1_ARB_RW1_RA[i],        USBFS_1_EP[ep].wBuffOffset & 0xFFu);
        CY_SET_REG8(&USBFS_1_ARB_RW1_RA_MSB[i],    (USBFS_1_EP[ep].wBuffOffset >> 8));
        CY_SET_REG8(&USBFS_1_ARB_RW1_WA[i],        USBFS_1_EP[ep].wBuffOffset & 0xFFu);
        CY_SET_REG8(&USBFS_1_ARB_RW1_WA_MSB[i],    (USBFS_1_EP[ep].wBuffOffset >> 8));
        
        ep++;
    }
    CY_SET_REG8(&USBFS_1_SIE_EP_INT_EN, 0xFF);
}
/*******************************************************************************
* Function Name: USBFS_1_GetConfigTablePtr
********************************************************************************
* Summary:
*   This routine returns a pointer a configuration table entry
*   
* Parameters:  
*   c Configuration Index
*******************************************************************************/
T_USBFS_1_LUT *USBFS_1_GetConfigTablePtr(uint8 c)
{
    uint8 cfg;
    /* Device Table */
    T_USBFS_1_LUT *pTmp = USBFS_1_GetDeviceTablePtr();

    /* The first entry points to the Device Descriptor.  */
    pTmp++;
    /* Top of the Configuration Table List*/
    pTmp = pTmp->p_list;

    /* Now get the config */
    for (cfg = c; cfg > 0; cfg--)
    {
        pTmp++;
    }
    return pTmp;
}
/*******************************************************************************
* Function Name: USBFS_1_GetDeviceTablePtr
********************************************************************************
* Summary:
*   This routine returns a pointer to the current config table based on the
*   selector in the request wValueLo
*   
* Parameters:  
*   None
*******************************************************************************/
T_USBFS_1_LUT *USBFS_1_GetDeviceTablePtr()
{
    /* Device Table */
    return USBFS_1_TABLE[USBFS_1_bDevice].p_list;
}

/*******************************************************************************
* Function Name: USBFS_1_SetEndpointHalt
********************************************************************************
* Summary:
*   This routine handles set endpoint halt
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_1_SetEndpointHalt()
{
    uint8 ep, ri;
    uint8 bRequestHandled = USBFS_1_FALSE;

    /* Clear endpoint halt */
    ep = (CY_GET_REG8(USBFS_1_wIndexLo) & 0x07);
    ri = ((ep - 1) << 4);
    if (ep != 0)
    {
        /* Set the endpoint Halt */
        USBFS_1_EP[ep].bHWEPState |= (USBFS_1_ENDPOINT_STATUS_HALT);
        
        /* Clear the data toggle */
        USBFS_1_EP[ep].bEPToggle = 0;
        
        if (USBFS_1_EP[ep].bAddr & USBFS_1_DIR_IN)
        {
            /* IN Endpoint */
            USBFS_1_EP[ep].bAPIEPState = USBFS_1_NO_EVENT_ALLOWED;
            CY_SET_REG8(&USBFS_1_SIE_EP1_CR0[ri], (USBFS_1_MODE_STALL_DATA_EP | USBFS_1_MODE_ACK_IN));
        }
        else
        {
            /* OUT Endpoint */
            USBFS_1_EP[ep].bAPIEPState = USBFS_1_NO_EVENT_ALLOWED;
            CY_SET_REG8(&USBFS_1_SIE_EP1_CR0[ri], (USBFS_1_MODE_STALL_DATA_EP | USBFS_1_MODE_ACK_OUT));
        }
        bRequestHandled = USBFS_1_InitNoDataControlTransfer();
    }
    return bRequestHandled;
}
/*******************************************************************************
* Function Name: USBFS_1_ClearEndpointHalt
********************************************************************************
* Summary:
*   This routine handles clear endpoint halt
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_1_ClearEndpointHalt()
{
    uint8 ep, ri;
    uint8 bRequestHandled = USBFS_1_FALSE;

    /* Clear endpoint halt */
    ep = (CY_GET_REG8(USBFS_1_wIndexLo) & 0x07);
    ri = ((ep  - 1) << 4);

    if (ep != 0)
    {
        /* Set the endpoint Halt */
        USBFS_1_EP[ep].bHWEPState &= ~(USBFS_1_ENDPOINT_STATUS_HALT);
        
        /* Clear the data toggle */
        USBFS_1_EP[ep].bEPToggle = 0;
        
        if (USBFS_1_EP[ep].bAddr & USBFS_1_DIR_IN)
        {
            /* IN Endpoint */
            USBFS_1_EP[ep].bAPIEPState = USBFS_1_NO_EVENT_PENDING;
            CY_SET_REG8(&USBFS_1_SIE_EP1_CR0[ri], USBFS_1_MODE_NAK_IN);
        }
        else
        {
            /* OUT Endpoint */
            USBFS_1_EP[ep].bAPIEPState = USBFS_1_EVENT_PENDING;
            CY_SET_REG8(&USBFS_1_SIE_EP1_CR0[ri], USBFS_1_MODE_ACK_OUT);
        }
        bRequestHandled = USBFS_1_InitNoDataControlTransfer();
    }
    return bRequestHandled;
}