/*******************************************************************************
* File Name: USBFS.c
* Version 1.30
*
*  Description:
*    API for USBFS Component.
*
*   Note:
*   
*   Many of the functions use endpoint number.  RAM arrays are sized with 9
*   elements so they are indexed directly by bEPNumber.  The SIE and ARB
*   registers are indexed by variations of bEPNumber - 1.
*
********************************************************************************
* Copyright 2008-2009, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
********************************************************************************/



#include "cydevice_trm.h"
#include "cyfitter.h"
#include "CyLib.h"
#include "USBFS.h"

/*******************************************************************************
* External data references
********************************************************************************/
extern uint8 USBFS_bConfiguration;
extern uint8 USBFS_bInterfaceSetting[];
extern uint8 USBFS_bDeviceAddress;
extern uint8 USBFS_bDeviceStatus;
extern uint8 USBFS_bEPHalt;
extern uint8 USBFS_bDevice;
extern uint8 USBFS_bTransferState;
extern T_USBFS_EP_CTL_BLOCK USBFS_EP[];

/*******************************************************************************
* Forward function references
********************************************************************************/
void USBFS_InitComponent(uint8 bDevice, uint8 bMode);

/*******************************************************************************
* External function references
********************************************************************************/
CY_ISR_PROTO(USBFS_EP_0_ISR);
#if(USBFS_EP1_ISR_REMOVE == 0)
CY_ISR_PROTO(USBFS_EP_1_ISR);
#endif   
#if(USBFS_EP2_ISR_REMOVE == 0)
CY_ISR_PROTO(USBFS_EP_2_ISR);
#endif   
#if(USBFS_EP3_ISR_REMOVE == 0)
CY_ISR_PROTO(USBFS_EP_3_ISR);
#endif   
#if(USBFS_EP4_ISR_REMOVE == 0)
CY_ISR_PROTO(USBFS_EP_4_ISR);
#endif   
#if(USBFS_EP5_ISR_REMOVE == 0)
CY_ISR_PROTO(USBFS_EP_5_ISR);
#endif   
#if(USBFS_EP6_ISR_REMOVE == 0)
CY_ISR_PROTO(USBFS_EP_6_ISR);
#endif   
#if(USBFS_EP7_ISR_REMOVE == 0)
CY_ISR_PROTO(USBFS_EP_7_ISR);
#endif   
#if(USBFS_EP8_ISR_REMOVE == 0)
CY_ISR_PROTO(USBFS_EP_8_ISR);
#endif   
CY_ISR_PROTO(USBFS_BUS_RESET_ISR);
CY_ISR_PROTO(USBFS_SOF_ISR);

/*******************************************************************************
* Function Name: USBFS_Start
********************************************************************************
* Summary:
*   This function initialize the USB SIE, arbiter and the
*   endpoint APIs, including setting the D+ Pullup
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_Start(uint8 bDevice, uint8 bMode)
{
    uint16 i;
    uint8 *p = (uint8 *)&USBFS_ARB_RW1_DR[0];
    
    /* Enable USB block */
    CY_SET_REG8(CYDEV_PM_ACT_CFG5, CY_GET_REG8(CYDEV_PM_ACT_CFG5) | 0x01);
    //TODO: pbvr: disable PM available for USB
    CY_SET_REG8(CYDEV_PM_AVAIL_CR6, CY_GET_REG8(CYDEV_PM_AVAIL_CR6) & ~0x10);
    /* Enable core clock */
    CY_SET_REG8(USBFS_USB_CLK_EN, 0x01);
    
    /* Bus Reset Length */
    CY_SET_REG8(USBFS_BUS_RST_CNT, 3);
    // TODO: pbvr: Enable PM available for USB
    CY_SET_REG8(CYDEV_PM_AVAIL_CR6, CY_GET_REG8(CYDEV_PM_AVAIL_CR6) | 0x10);
    /* If VBUS Monitoring is enable, setup the DR in the port reg */
#if (USBFS_MON_VBUS == 1)
    CY_SET_REG8(USBFS_VBUS_DR, CY_GET_REG8(USBFS_VBUS_DR) & ~USBFS_VBUS_MASK);
#endif

    /* Write WAx */
    CY_SET_REG8(&USBFS_ARB_RW1_WA[0],     0);
    CY_SET_REG8(&USBFS_ARB_RW1_WA_MSB[0], 0);
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < 512; i++)
    {
        CY_SET_REG8(p, 0xFF);
    }

    /* Set the bus reset Interrupt. */
    CyIntSetVector(USBFS_BUS_RESET_VECT_NUM,   USBFS_BUS_RESET_ISR);
    CyIntSetPriority(USBFS_BUS_RESET_VECT_NUM, USBFS_BUS_RESET_PRIOR);
    CyIntEnable(USBFS_BUS_RESET_VECT_NUM);

    /* Set the SOF Interrupt. */
    CyIntSetVector(USBFS_SOF_VECT_NUM,   USBFS_SOF_ISR);
    CyIntSetPriority(USBFS_SOF_VECT_NUM, USBFS_SOF_PRIOR);
    CyIntEnable(USBFS_SOF_VECT_NUM);

    /* Set the Control Endpoint Interrupt. */
    CyIntSetVector(USBFS_EP_0_VECT_NUM,   USBFS_EP_0_ISR);
    CyIntSetPriority(USBFS_EP_0_VECT_NUM, USBFS_EP_0_PRIOR);
    CyIntEnable(USBFS_EP_0_VECT_NUM);

    /* Set the Data Endpoint 1 Interrupt. */
#if(USBFS_EP1_ISR_REMOVE == 0)
    CyIntSetVector(USBFS_EP_1_VECT_NUM,   USBFS_EP_1_ISR);
    CyIntSetPriority(USBFS_EP_1_VECT_NUM, USBFS_EP_1_PRIOR);
    CyIntEnable(USBFS_EP_1_VECT_NUM);
#endif   // End USBFS_EP1_ISR_REMOVE

    /* Set the Data Endpoint 2 Interrupt. */
#if(USBFS_EP2_ISR_REMOVE == 0)
    CyIntSetVector(USBFS_EP_2_VECT_NUM,   USBFS_EP_2_ISR);
    CyIntSetPriority(USBFS_EP_2_VECT_NUM, USBFS_EP_2_PRIOR);
    CyIntEnable(USBFS_EP_2_VECT_NUM);
#endif   // End USBFS_EP2_ISR_REMOVE

    /* Set the Data Endpoint 3 Interrupt. */
#if(USBFS_EP3_ISR_REMOVE == 0)
    CyIntSetVector(USBFS_EP_3_VECT_NUM,   USBFS_EP_3_ISR);
    CyIntSetPriority(USBFS_EP_3_VECT_NUM, USBFS_EP_3_PRIOR);
    CyIntEnable(USBFS_EP_3_VECT_NUM);
#endif   // End USBFS_EP3_ISR_REMOVE

    /* Set the Data Endpoint 4 Interrupt. */
#if(USBFS_EP4_ISR_REMOVE == 0)
    CyIntSetVector(USBFS_EP_4_VECT_NUM,   USBFS_EP_4_ISR);
    CyIntSetPriority(USBFS_EP_4_VECT_NUM, USBFS_EP_4_PRIOR);
    CyIntEnable(USBFS_EP_4_VECT_NUM);
#endif   // End USBFS_EP4_ISR_REMOVE

    /* Set the Data Endpoint 5 Interrupt. */
#if(USBFS_EP5_ISR_REMOVE == 0)
    CyIntSetVector(USBFS_EP_5_VECT_NUM,   USBFS_EP_5_ISR);
    CyIntSetPriority(USBFS_EP_5_VECT_NUM, USBFS_EP_5_PRIOR);
    CyIntEnable(USBFS_EP_5_VECT_NUM);
#endif   // End USBFS_EP5_ISR_REMOVE

    /* Set the Data Endpoint 6 Interrupt. */
#if(USBFS_EP6_ISR_REMOVE == 0)
    CyIntSetVector(USBFS_EP_6_VECT_NUM,   USBFS_EP_6_ISR);
    CyIntSetPriority(USBFS_EP_6_VECT_NUM, USBFS_EP_6_PRIOR);
    CyIntEnable(USBFS_EP_6_VECT_NUM);
#endif   // End USBFS_EP6_ISR_REMOVE

     /* Set the Data Endpoint 7 Interrupt. */
#if(USBFS_EP7_ISR_REMOVE == 0)
    CyIntSetVector(USBFS_EP_7_VECT_NUM,   USBFS_EP_7_ISR);
    CyIntSetPriority(USBFS_EP_7_VECT_NUM, USBFS_EP_7_PRIOR);
    CyIntEnable(USBFS_EP_7_VECT_NUM);
#endif   // End USBFS_EP7_ISR_REMOVE

    /* Set the Data Endpoint 8 Interrupt. */
#if(USBFS_EP8_ISR_REMOVE == 0)
    CyIntSetVector(USBFS_EP_8_VECT_NUM,   USBFS_EP_8_ISR);
    CyIntSetPriority(USBFS_EP_8_VECT_NUM, USBFS_EP_8_PRIOR);
    CyIntEnable(USBFS_EP_8_VECT_NUM);
#endif   // End USBFS_EP8_ISR_REMOVE
    
    USBFS_InitComponent(bDevice, bMode);
}
/*******************************************************************************
* Function Name: USBFS_InitComponent
********************************************************************************
* Summary:
*   Initialize the component, except for the HW which is done one time in
*	the Start function.  This function pulls up D+.
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_InitComponent(uint8 bDevice, uint8 bMode)
{
    /* USB Locking: Enabled, VRegulator: Disabled */
    CY_SET_REG8(USBFS_CR1, (bMode | 0x02));

    /* Record the descriptor selection */
    USBFS_bDevice = bDevice;

    /* Clear all of the component data */
    USBFS_bConfiguration = 0;
    USBFS_bDeviceAddress  = 0;
    USBFS_bEPHalt = 0;
    USBFS_bDeviceStatus = 0;
    USBFS_bDeviceStatus = 0;
	/*TODO Add hid var*/
    USBFS_bTransferState = USBFS_TRANS_STATE_IDLE;

    /* STALL_IN_OUT */
    CY_SET_REG8(USBFS_EP0_CR, USBFS_MODE_STALL_IN_OUT);
    /* Enable the SIE with an address 0 */
    CY_SET_REG8(USBFS_CR0, USBFS_CR0_ENABLE );
    /* Finally, Enable d+ pullup */
    CY_SET_REG8(USBFS_USBIO_CR1, USBFS_USBIO_CR1_USBPUEN);
}
/*******************************************************************************
* Function Name: USBFS_Stop
********************************************************************************
* Summary:
*   This function shuts down the USB function including to release
*   the D+ Pullup and disabling the SIE.
*
* Parameters:  
*   None
*******************************************************************************/
void USBFS_Stop(void)
{
    /* Disable the SIE with address 0 */
    CY_SET_REG8(USBFS_CR0, 0x00);
    /* Disable the d+ pullup */
    CY_SET_REG8(USBFS_USBIO_CR1, 0x00);
    /* Disable the reset interrupt */
    CyIntDisable(USBFS_BUS_RESET_VECT_NUM);
    CyIntDisable(USBFS_EP_0_VECT_NUM);

    USBFS_bConfiguration = 0;
    USBFS_bDeviceAddress  = 0;
    USBFS_bEPHalt = 0;
    USBFS_bDeviceStatus = 0;
}
/*******************************************************************************
* Function Name: USBFS_bCheckActivity
********************************************************************************
* Summary:
*   Returns the activity status of the bus.  Clears the status hardware to
*   provide fresh activity status on the next call of this routine.
*
* Parameters:  
*   None
*******************************************************************************/
uint8  USBFS_bCheckActivity(void)
{
    uint8 r = ((CY_GET_REG8(USBFS_CR1) >> 2) & USBFS_CR1_BUS_ACTIVITY);

    CY_SET_REG8(USBFS_CR1, (CY_GET_REG8(USBFS_CR1) & USBFS_CR1_BUS_ACTIVITY));

    return r;
}
/*******************************************************************************
* Function Name: USBFS_bGetConfiguration
********************************************************************************
* Summary:
*   Returns the current configuration setting
*
* Parameters:  
*   None
*******************************************************************************/
uint8  USBFS_bGetConfiguration(void)
{
    return USBFS_bConfiguration;
}
/*******************************************************************************
* Function Name: USBFS_bGetInterfaceSetting
********************************************************************************
* Summary:
*   Returns the current interface setting
*
* Parameters:  
*   uint8 ifc, interface number
*
*******************************************************************************/
uint8  USBFS_bGetInterfaceSetting(uint8 ifc)
{
    return USBFS_bInterfaceSetting[ifc];
}
/*******************************************************************************
* Function Name: USBFS_bGetEPState
********************************************************************************
* Summary:
*   Returned the state of the requested endpoint.
*
* Parameters:  
*   bEPNumber: Endpoint Number
*******************************************************************************/
uint8 USBFS_bGetEPState(uint8 bEPNumber)
{
    return USBFS_EP[bEPNumber].bAPIEPState;
}
/*******************************************************************************
* Function Name: USBFS_wGetEPCount
********************************************************************************
* Summary:
*   Returns the transfer count for the requested endpoint.  The value from
*   the count registers includes 2 counts for the two byte checksum of the
*   packet.  This function subtracts the two counts.
*
* Parameters:  
*   bEPNumber: Endpoint Number
*******************************************************************************/
uint16 USBFS_wGetEPCount(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);
    return ( (uint16)(CY_GET_REG8(&USBFS_SIE_EP1_CNT0[ri]) & USBFS_EPX_CNT0_MASK) << 8 | 
                      CY_GET_REG8(&USBFS_SIE_EP1_CNT1[ri]) - 2 );

}
/*******************************************************************************
* Function Name: USBFS_LoadEP
********************************************************************************
* Summary:  Load the endpoint buffer, set up the address pointers and go.
*
* Parameters:  
*   None
*******************************************************************************/
void   USBFS_LoadEP(uint8 bEPNumber, uint8 *pData, uint16 wLength)
{
    uint8 i;
    uint8 ri = ((bEPNumber - 1) << 4);
    uint8 *p = (uint8 *)&USBFS_ARB_RW1_DR[ri];
    
    /* Write WAx */
    CY_SET_REG8(&USBFS_ARB_RW1_WA[ri],     USBFS_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&USBFS_ARB_RW1_WA_MSB[ri], (USBFS_EP[bEPNumber].wBuffOffset >> 8));
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < wLength; i++)
    {
        CY_SET_REG8(p, *pData++);
    }
    /* Set the count and data toggle */
    CY_SET_REG8(&USBFS_SIE_EP1_CNT0[ri], (wLength >> 8) | (USBFS_EP[bEPNumber].bEPToggle));
    CY_SET_REG8(&USBFS_SIE_EP1_CNT1[ri],  wLength & 0xFFu);
    /* Write the RAx */
    CY_SET_REG8(&USBFS_ARB_RW1_RA[ri],     USBFS_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&USBFS_ARB_RW1_RA_MSB[ri], (USBFS_EP[bEPNumber].wBuffOffset >> 8));

    /* Mark the event pending */
    USBFS_EP[bEPNumber].bAPIEPState = USBFS_NO_EVENT_PENDING;
    /* Write the Mode register */
    CY_SET_REG8(&USBFS_SIE_EP1_CR0[ri], USBFS_EP[bEPNumber].bEPMode);
}

/*******************************************************************************
* Function Name: USBFS_ReadOutEP
********************************************************************************
* Summary:
*   Read data from an endpoint.  The application must call USBFS_bGetEPState
*   to see if an event is pending.
*
* Parameters:  
*   bEPNumber   Endpoint Number
*   pData       Pointer to the destination buffer
*   wLength     Length of the destination buffer
*
* Returns:
*   Number of bytes received
*
*******************************************************************************/
uint16 USBFS_ReadOutEP(uint8 bEPNumber, uint8 *pData, uint16 wLength)
{
    uint8 i;
    uint8 ri = ((bEPNumber - 1) << 4);
    uint8 *p = (uint8 *)&USBFS_ARB_RW1_DR[ri];
/*
	USB_bEP0Count = (CY_GET_REG8(USB_EP0_CNT ) & 0x0F) - 2;
    
    USB_TransferByteCount += USB_bEP0Count;
    
    while ((CurrentTD.wCount > 0) && (USB_bEP0Count > 0))
	{
        *CurrentTD.pData++ = CY_GET_REG8(pReg++);
        USB_bEP0Count--;
		CurrentTD.wCount--;
	}*/
    
    /* Write the RAx */
    CY_SET_REG8(&USBFS_ARB_RW1_RA[ri],     USBFS_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&USBFS_ARB_RW1_RA_MSB[ri], (USBFS_EP[bEPNumber].wBuffOffset >> 8));
    
    /* Copy the data using the arbiter data register */
    for (i = 0; i < wLength; i++)
    {
        *pData++ = CY_GET_REG8(p);
    }
    /* Write the WAx */
    CY_SET_REG8(&USBFS_ARB_RW1_WA[ri],     USBFS_EP[bEPNumber].wBuffOffset & 0xFFu);
    CY_SET_REG8(&USBFS_ARB_RW1_WA_MSB[ri], (USBFS_EP[bEPNumber].wBuffOffset >> 8));

    /* (re)arming of OUT endpoint */
    USBFS_EnableOutEP(bEPNumber);

    return wLength;
}
/*******************************************************************************
* Function Name: USBFS_EnableOutEP
********************************************************************************
* Summary:
*    Enable an OUT endpoint
*
* Parameters:  
*   bEPNumber   Endpoint Number
*
*******************************************************************************/
void USBFS_EnableOutEP(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);
    /* Mark the event pending */
    USBFS_EP[bEPNumber].bAPIEPState = USBFS_NO_EVENT_PENDING;
    /* Write the Mode register */
    CY_SET_REG8(&USBFS_SIE_EP1_CR0[ri], USBFS_EP[bEPNumber].bEPMode);
}
/*******************************************************************************
* Function Name: USBFS_DisableOutEP
********************************************************************************
* Summary:
*
* Parameters:  
*   bEPNumber   Endpoint Number
*
*******************************************************************************/
void USBFS_DisableOutEP(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);

	/* Write the Mode register */
    CY_SET_REG8(&USBFS_SIE_EP1_CR0[ri], USBFS_EP[bEPNumber].bEPMode);
}
/*******************************************************************************
* Function Name: USBFS_Force
********************************************************************************
* Summary:  Forces the bus state
*
* Parameters:  
*   bState 
*    USBFS_FORCE_J 
*    USBFS_FORCE_K 
*    USBFS_FORCE_SE0 
*    USBFS_FORCE_NONE
*
*******************************************************************************/
void USBFS_Force(uint8 bState)
{
    CY_SET_REG8(USBFS_USBIO_CR0, bState);
}
/*******************************************************************************
* Function Name: USBFS_bGetEPAckState
********************************************************************************
* Summary:  Returns the ACK of the CR0 Register (ACKD)
*
* Parameters:  
*   bEPNumber   Endpoint Number
*
* Returns
*   0 if nothing has been ACKD, non-=zero something has been ACKD
*******************************************************************************/
uint8 USBFS_bGetEPAckState(uint8 bEPNumber)
{
    uint8 ri = ((bEPNumber - 1) << 4);
  
    return (CY_GET_REG8(&USBFS_SIE_EP1_CR0[ri]) & USBFS_MODE_ACKD );
}
/*******************************************************************************
* Function Name: USBFS_SetPowerStatus
********************************************************************************
* Summary:
*   Sets the device power status for reporting in the Get Device Status
*   request
*
* Parameters:  
*   bPowerStaus 0 = Bus Powered, 1 = Self Powered
*
*******************************************************************************/
void USBFS_SetPowerStatus(uint8 bPowerStaus)
{
    if (bPowerStaus)
    {
        USBFS_bDeviceStatus |=  USBFS_DEVICE_STATUS_SELF_POWERED;
    }
    else
    {
        USBFS_bDeviceStatus &=  ~USBFS_DEVICE_STATUS_SELF_POWERED;
    }
}

/*******************************************************************************
* Function Name: USBFS_bVBusPresent
********************************************************************************
* Summary:
*   Determines VBUS presense for Self Powered Devices.  Returns 1 if VBUS
*   is present, otherwise 0.
*
* Parameters:  
*   None
*******************************************************************************/
#if (USBFS_MON_VBUS == 1)
uint8 USBFS_bVBusPresent()
{
    return ((CY_GET_REG8(USBFS_VBUS_PS) & USBFS_VBUS_MASK) ? 1 : 0);
}
#endif

