/*******************************************************************************
* File Name: USBFS_drv.c
* Version 1.30
*
*  Description:
*    Endpoint 0 Driver for the USBFS Component.
*
*   Note:
*
********************************************************************************
* Copyright 2008-2009, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
********************************************************************************/



#include "cydevice_trm.h"
#include "cyfitter.h"
#include "USBFS.h"

/*******************************************************************************
* Forward references for the EP0 ISR
********************************************************************************/
void  USBFS_HandleSetup(void);
void  USBFS_HandleIN(void);
void  USBFS_HandleOUT(void);
uint8 USBFS_InitControlRead(void);
void  USBFS_ControlReadDataStage(void);
void  USBFS_ControlReadStatusStage(void);
void  USBFS_ControlReadPrematureStatus(void);
uint8 USBFS_InitControlWrite(void);
void  USBFS_ControlWriteDataStage(void);
void  USBFS_ControlWriteStatusStage(void);
void  USBFS_ControlWritePrematureStatus(void);
uint8 USBFS_InitNoDataControlTransfer(void);
void  USBFS_NoDataControlStatusStage(void);
void USBFS_InitializeStatusBlock(void);
void USBFS_UpdateStatusBlock(uint8 bCompletionCode);

/*******************************************************************************
* Request Handlers
********************************************************************************/
uint8 USBFS_HandleStandardRqst(void);
uint8 USBFS_DispatchClassRqst(void);
uint8 USBFS_HandleVendorRqst(void);

/*******************************************************************************
* External data references
********************************************************************************/

/*******************************************************************************
* Global data allocation
********************************************************************************/
T_USBFS_EP_CTL_BLOCK USBFS_EP[9];
uint8 USBFS_bEPHalt;
uint8 USBFS_bConfiguration;
uint8 USBFS_bInterfaceSetting[9];
uint8 USBFS_bDeviceAddress;
uint8 USBFS_bDeviceStatus;
uint8 USBFS_bDevice;
/*******************************************************************************
* Local data allocation
********************************************************************************/
uint8 USBFS_bEP0Toggle;
uint8 USBFS_bLastPacketSize;
uint8 USBFS_bTransferState;
T_USBFS_TD CurrentTD;
uint8 USBFS_bEP0Mode;
uint8 USBFS_bEP0Count;
uint16 USBFS_TransferByteCount;

/*******************************************************************************
* Function Name: USBFS_ep_0_Interrupt
********************************************************************************
* Summary:
*   This Interrupt Service Routine handles Endpoint 0 (Control Pipe) traffic.  It
*   dispactches setup requests and handles the data and status stages.
*   
* Parameters:  
*   None
*******************************************************************************/
CY_ISR(USBFS_EP_0_ISR)
{
	uint8 bRegTemp = *USBFS_EP0_CR;
		
	if (!bRegTemp & USBFS_MODE_ACKD) return;             
	if (bRegTemp & USBFS_MODE_SETUP_RCVD) {
		USBFS_HandleSetup();
	} 
	else if (bRegTemp & USBFS_MODE_IN_RCVD) {
		USBFS_HandleIN();	
	}
	else if (bRegTemp & USBFS_MODE_OUT_RCVD) {
		USBFS_HandleOUT();
	}
	else {
//		ASSERT(0);
	}
	CY_SET_REG8(USBFS_EP0_CNT, USBFS_bEP0Toggle | USBFS_bEP0Count);
	/* Set the Mode Register  */
	CY_SET_REG8(USBFS_EP0_CR, USBFS_bEP0Mode);
	
   /* PSoC3 ES1, ES2 RTC ISR PATCH  */ 
    #if(CYDEV_CHIP_DIE_EXPECT == CYDEV_CHIP_DIE_LEOPARD)
        #if((CYDEV_CHIP_REV_EXPECT <= CYDEV_CHIP_REV_LEOPARD_ES2) && (USBFS_ep_0__ES2_PATCH))      
            USBFS_ISR_PATCH();
        #endif
    #endif	
	
}
/*******************************************************************************
* Function Name: USBFS_HandleSetup
********************************************************************************
* Summary:
*   This Routine dispatches requests for the four USB request types
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_HandleSetup(void)
{
	uint8 bRequestHandled;
    /* In case the previous transfer did not complete, close it out */
    USBFS_UpdateStatusBlock(USBFS_XFER_PREMATURE);
    
    switch (CY_GET_REG8(USBFS_bmRequestType) & USBFS_RQST_TYPE_MASK)
	{
	case USBFS_RQST_TYPE_STD:
		bRequestHandled = USBFS_HandleStandardRqst();
		break;
	case USBFS_RQST_TYPE_CLS:
		bRequestHandled = USBFS_DispatchClassRqst();
		break;
	case USBFS_RQST_TYPE_VND:
		bRequestHandled = USBFS_HandleVendorRqst();
		break;
	default:
		bRequestHandled = USBFS_FALSE;
		break;
	}
	if (bRequestHandled == USBFS_FALSE){
		USBFS_bEP0Mode = USBFS_MODE_STALL_IN_OUT;
	}
}
/*******************************************************************************
* Function Name: USBFS_HandleIN
********************************************************************************
* Summary:
*   This routine handles EP0 IN transfers.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_HandleIN(void)
{
	switch (USBFS_bTransferState)
	{
	case USBFS_TRANS_STATE_IDLE:
		break;
	case USBFS_TRANS_STATE_CONTROL_READ:
		USBFS_ControlReadDataStage();
		break;
	case USBFS_TRANS_STATE_CONTROL_WRITE:
		USBFS_ControlReadStatusStage();
		break;
	case USBFS_TRANS_STATE_NO_DATA_CONTROL:
		USBFS_NoDataControlStatusStage();
		break;
	default:
		break;	
	}
}
/*******************************************************************************
* Function Name: USBFS_HandleOUT
********************************************************************************
* Summary:
*   This routine handles EP0 OUT transfers.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_HandleOUT(void)
{
	switch (USBFS_bTransferState)
	{
	case USBFS_TRANS_STATE_IDLE:
		break;
	case USBFS_TRANS_STATE_CONTROL_READ:
		USBFS_ControlReadStatusStage();
		break;
	case USBFS_TRANS_STATE_CONTROL_WRITE:
		USBFS_ControlWriteDataStage();
		break;
	case USBFS_TRANS_STATE_NO_DATA_CONTROL:
        /* Update the completion block */
        USBFS_UpdateStatusBlock(USBFS_XFER_ERROR);
        /* We expect no more data, so stall INs and OUTs */
        USBFS_bEP0Mode = USBFS_MODE_STALL_IN_OUT;
		break;
	default:
		break;	
	}
}
/*******************************************************************************
* Function Name: USBFS_LoadEP0
********************************************************************************
* Summary:
*   This routine loads the EP0 data registers for OUT transfers.  It uses the
*   CurrentTD (previously initialized by the _InitControlWrite function and
*   updated for each OUT transfer, and the bLastPacketSize) to determine how
*   many uint8s to transfer on the current OUT.
*
*   If the number of uint8s remaining is zero and the last transfer was full, 
*   we need to send a zero length packet.  Otherwise we send the minimum
*   of the control endpoint size (8) or remaining number of uint8s for the
*   transaction.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_LoadEP0(void)
{
	/* Update the transfer byte count from the last transaction */
    USBFS_TransferByteCount += USBFS_bLastPacketSize;
    /* Now load the next transaction */
    USBFS_bEP0Count = 0;
	while ((CurrentTD.wCount > 0) && (USBFS_bEP0Count < 8))
	{
        USBFS_EP0_DR0[USBFS_bEP0Count] = *CurrentTD.pData++;
		USBFS_bEP0Count++;
		CurrentTD.wCount--;
	}

	if ((USBFS_bEP0Count > 0) || (USBFS_bLastPacketSize == 8))
	{
		/* Update the data toggle */
		USBFS_bEP0Toggle ^= USBFS_EP0_CNT_DATA_TOGGLE;
		/* Set the Mode Register  */
		USBFS_bEP0Mode = USBFS_MODE_ACK_IN_STATUS_OUT;
		/* Update the state (or stay the same) */
		USBFS_bTransferState = USBFS_TRANS_STATE_CONTROL_READ;
	}
	else
	{
		/* Expect Status Stage Out */
		USBFS_bEP0Mode = USBFS_MODE_STATUS_OUT_ONLY;
		/* Update the state (or stay the same) */
		USBFS_bTransferState = USBFS_TRANS_STATE_CONTROL_READ;
	}

	/* Save the packet size for next time */
	USBFS_bLastPacketSize = USBFS_bEP0Count;
}
/*******************************************************************************
* Function Name: USBFS_InitControlRead
********************************************************************************
* Summary:
*   Initialize a control read transaction
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_InitControlRead(void)
{
    uint16 wXferCount;

    /* Set up the state machine */
    USBFS_bTransferState = USBFS_TRANS_STATE_CONTROL_READ;
    /* Set the toggle, it gets updated in LoadEP */
    USBFS_bEP0Toggle = 0;
    /* Initialize the Status Block */
    USBFS_InitializeStatusBlock();
    wXferCount = (((uint16)*USBFS_wLengthHi << 8) | (*USBFS_wLengthLo));

    if (CurrentTD.wCount > wXferCount)
    {
        CurrentTD.wCount = wXferCount;
    }
    USBFS_LoadEP0();

    return USBFS_TRUE;
}
/*******************************************************************************
* Function Name: USBFS_ControlReadDataStage
********************************************************************************
* Summary:
*   Handle the Data Stage of a control read transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_ControlReadDataStage(void)
{
	USBFS_LoadEP0();
}
/*******************************************************************************
* Function Name: USBFS_ControlReadStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a control read transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_ControlReadStatusStage(void)
{
	/* Update the transfer byte count */
    USBFS_TransferByteCount += USBFS_bLastPacketSize;
    /* Go Idle */
    USBFS_bTransferState = USBFS_TRANS_STATE_IDLE;
    /* Update the completion block */
    USBFS_UpdateStatusBlock(USBFS_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	USBFS_bEP0Mode = USBFS_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: USBFS_InitControlWrite
********************************************************************************
* Summary:
*   Initialize a control write transaction
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_InitControlWrite(void)
{
    uint16 wXferCount;
    
    /* Set up the state machine */
    USBFS_bTransferState = USBFS_TRANS_STATE_CONTROL_WRITE;
    /* This migh not be necessary */
    USBFS_bEP0Toggle = USBFS_EP0_CNT_DATA_TOGGLE;;
    /* Initialize the Status Block */
    USBFS_InitializeStatusBlock();

    wXferCount = (((uint16)CY_GET_REG8(USBFS_wLengthHi) << 8) | (CY_GET_REG8(USBFS_wLengthLo)));

    if (CurrentTD.wCount > wXferCount)
    {
        CurrentTD.wCount = wXferCount;
    }
    
	/* Expect Data or Status Stage */
	USBFS_bEP0Mode = USBFS_MODE_ACK_OUT_STATUS_IN;
    
    return USBFS_TRUE;
}
/*******************************************************************************
* Function Name: USBFS_ControlWriteDataStage
********************************************************************************
* Summary:
*   Handle the Data Stage of a control write transfer
*       1. Get the data (We assume the destination was validated previously)
*       3. Update the count and data toggle
*       2. Update the mode register for the next transaction
* Parameters:  
*   None
*******************************************************************************/
void USBFS_ControlWriteDataStage(void)
{
	uint8 *pReg = (uint8 *)USBFS_EP0_DR0; 
	USBFS_bEP0Count = (CY_GET_REG8(USBFS_EP0_CNT ) & 0x0F) - 2;
    
    USBFS_TransferByteCount += USBFS_bEP0Count;
    
    while ((CurrentTD.wCount > 0) && (USBFS_bEP0Count > 0))
	{
        *CurrentTD.pData++ = CY_GET_REG8(pReg++);
        USBFS_bEP0Count--;
		CurrentTD.wCount--;
	}
	/* Update the data toggle */
	USBFS_bEP0Toggle ^= USBFS_EP0_CNT_DATA_TOGGLE;
	/* Expect Data or Status Stage */
	USBFS_bEP0Mode = USBFS_MODE_ACK_OUT_STATUS_IN;
}
/*******************************************************************************
* Function Name: USBFS_ControlWriteStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a control write transfer
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_ControlWriteStatusStage(void)
{
	/* Go Idle */
    USBFS_bTransferState = USBFS_TRANS_STATE_IDLE;
    /* Update the completion block */
    USBFS_UpdateStatusBlock(USBFS_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	USBFS_bEP0Mode = USBFS_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: USBFS_InitNoDataControlTransfer
********************************************************************************
* Summary:
*   Initialize a no data control transfer
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_InitNoDataControlTransfer(void)
{
	USBFS_bTransferState = USBFS_TRANS_STATE_NO_DATA_CONTROL;
	USBFS_bEP0Mode = USBFS_MODE_STATUS_IN_ONLY;
	return USBFS_TRUE;
}
/*******************************************************************************
* Function Name: USBFS_NoDataControlStatusStage
********************************************************************************
* Summary:
*   Handle the Status Stage of a no data control transfer.
*
*   SET_ADDRESS is special, since we need to receive the status stage with
*   the old address.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_NoDataControlStatusStage(void)
{
	/* Change the USB address register if we got a SET_ADDRESS. */
    if (USBFS_bDeviceAddress != 0)
    {
        CY_SET_REG8(USBFS_CR0, USBFS_bDeviceAddress | USBFS_CR0_ENABLE);
        USBFS_bDeviceAddress = 0;
    }
    	/* Go Idle */
    USBFS_bTransferState = USBFS_TRANS_STATE_IDLE;
    /* Update the completion block */
    USBFS_UpdateStatusBlock(USBFS_XFER_STATUS_ACK);
 	/* We expect no more data, so stall INs and OUTs */
	USBFS_bEP0Mode = USBFS_MODE_STALL_IN_OUT;
}
/*******************************************************************************
* Function Name: USBFS_UpdateStatusBlock
********************************************************************************
* Summary:
*   Update the Completion Status Block for a Request.  The block is updated
*   with the completion code the USBFS_TransferByteCount.  The
*   StatusBlock Pointer  is set to NULL.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_UpdateStatusBlock(uint8 bCompletionCode)
{
    if (CurrentTD.pStatusBlock != USBFS_NULL)
    {
        CurrentTD.pStatusBlock->bStatus = bCompletionCode;
        CurrentTD.pStatusBlock->wLength = USBFS_TransferByteCount;
        CurrentTD.pStatusBlock = USBFS_NULL;
    }
}
/*******************************************************************************
* Function Name: USBFS_InitializeStatusBlock
********************************************************************************
* Summary:
*   Initialize the Completion Status Block for a Request.  The completion
*   code is set to USB_XFER_IDLE.
*
*   Also, initializes USBFS_TransferByteCount.  Save some space,
*   this is the only consumer.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_InitializeStatusBlock(void)
{
    USBFS_TransferByteCount = 0;
    if (CurrentTD.pStatusBlock != USBFS_NULL)
    {
        CurrentTD.pStatusBlock->bStatus = USBFS_XFER_IDLE;
        CurrentTD.pStatusBlock->wLength = 0;
    }
}
