/*******************************************************************************
* File Name: USBFS_std.c
* Version 1.30
*
*  Description:
*    USB Standard request handler.
*
*   Note:
*
********************************************************************************
* Copyright 2008-2009, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
********************************************************************************/



#include "cydevice_trm.h"
#include "cyfitter.h"
#include "USBFS.h"

/*******************************************************************************
* External references
********************************************************************************/
uint8 USBFS_InitControlRead(void);
uint8 USBFS_InitControlWrite(void);
uint8 USBFS_InitNoDataControlTransfer(void);
uint8 USBFS_DispatchClassRqst(void);

extern uint8 USBFS_CODE USBFS_DEVICE0_DESCR[];
extern uint8 USBFS_CODE USBFS_DEVICE0_CONFIGURATION0_DESCR[]; 
extern uint8 USBFS_CODE USBFS_STRING_DESCRIPTORS[];
extern uint8 USBFS_CODE USBFS_MSOS_DESCRIPTOR[];
extern uint8 USBFS_CODE USBFS_SN_STRING_DESCRIPTOR[];


extern uint8 USBFS_bEPHalt;
extern uint8 USBFS_bDevice;
extern uint8 USBFS_bConfiguration;
extern uint8 USBFS_bInterfaceSetting[];
extern uint8 USBFS_bDeviceAddress;
extern uint8 USBFS_bDeviceStatus;
extern T_USBFS_LUT USBFS_CODE USBFS_TABLE[];
extern T_USBFS_EP_CTL_BLOCK USBFS_EP[];
extern T_USBFS_TD CurrentTD;
uint8 tBuffer[2];
uint8 *USBFS_bFWSerialNumberStringDescriptor;
uint8 USBFS_bSNStringConfirm = 0;

/*DIE ID string descriptor for 8 bytes ID*/
#if defined(USBFS_ENABLE_IDSN_STRING)
void USBFS_ReadDieID(uint8 *descr);
uint8 USBFS_bIDSerialNumberStringDescriptor[0x22]={0x22,0x03};
#endif	

/*******************************************************************************
* Forward references
********************************************************************************/
void USBFS_Config(void);
T_USBFS_LUT *USBFS_GetConfigTablePtr(uint8 c);
T_USBFS_LUT *USBFS_GetDeviceTablePtr(void);
uint8 USBFS_ClearEndpointHalt(void);
uint8 USBFS_SetEndpointHalt(void);
uint8 USBFS_ValidateAlternateSetting(void);
//uint8 USBFS_GetDeviceStatus(void);

/*******************************************************************************
* Function Name: USBFS_SerialNumString
********************************************************************************
* Summary:
*   Application firmware may supply the source of the USB device descriptors serial 
*   number string during runtime.
*   
* Parameters:  
*   SNstring - pointer to string
*******************************************************************************/
void  USBFS_SerialNumString(uint8 *SNstring)
{
#if defined(USBFS_ENABLE_FWSN_STRING)
    USBFS_bFWSerialNumberStringDescriptor = SNstring;
	/* check descriptor validation */
	if( (USBFS_bFWSerialNumberStringDescriptor[0] > 1 ) &&   /* Descriptor Length */
	    (USBFS_bFWSerialNumberStringDescriptor[1] == 3) )    /* DescriptorType: STRING */    
	{
	    USBFS_bSNStringConfirm = 1;
	}
    else
    {
	    USBFS_bSNStringConfirm = 0;
    }
#else
    SNstring = SNstring;
#endif     
}

/*******************************************************************************
* Function Name: USBFS_HandleStandardRqst
********************************************************************************
* Summary:
*   This Routine dispatches standard requests
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_HandleStandardRqst()
{
    uint8 bRequestHandled = USBFS_FALSE;
#if defined(USBFS_ENABLE_STRINGS)
    uint8 *pStr;
  #if defined(USBFS_ENABLE_DESCRIPTOR_STRINGS)
    uint8 nStr;
  #endif
#endif
    uint16 wCount;
    
    T_USBFS_LUT *pTmp;
    CurrentTD.wCount = 0;
    
    if ((CY_GET_REG8(USBFS_bmRequestType) & USBFS_RQST_DIR_MASK) == USBFS_RQST_DIR_D2H)
    {
        /* Control Read */
        switch (CY_GET_REG8(USBFS_bRequest)) 
        {
        case USBFS_GET_DESCRIPTOR:
            if (CY_GET_REG8(USBFS_wValueHi) == USBFS_DESCR_DEVICE)  
            {
                pTmp = USBFS_GetDeviceTablePtr();
                CurrentTD.pData     = pTmp->p_list;
                CurrentTD.wCount    = 18;
                bRequestHandled     = USBFS_InitControlRead();
                }
            else if (CY_GET_REG8(USBFS_wValueHi) == USBFS_DESCR_CONFIG)  {
                pTmp = USBFS_GetConfigTablePtr(CY_GET_REG8(USBFS_wValueLo));
                CurrentTD.pData     = pTmp->p_list;
                wCount    = ((uint16)(CurrentTD.pData)[3] << 8) | (CurrentTD.pData)[2];
                CurrentTD.wCount    = wCount;
                bRequestHandled     = USBFS_InitControlRead();
            }
            #if defined(USBFS_ENABLE_STRINGS)
                else if (CY_GET_REG8(USBFS_wValueHi) == USBFS_DESCR_STRING)
                {
                /* Descriptor Strings*/	
                #if defined(USBFS_ENABLE_DESCRIPTOR_STRINGS)
                    nStr = 0;
                    pStr = &USBFS_STRING_DESCRIPTORS[0];
                    while ((CY_GET_REG8(USBFS_wValueLo) > nStr) && (*pStr != 0)){
                        pStr += *pStr;
                        nStr++;
                    };
                #endif /* End USBFS_ENABLE_DESCRIPTOR_STRINGS */
                /* Microsoft OS String*/	
                #if defined(USBFS_ENABLE_MSOS_STRING)
                    if( CY_GET_REG8(USBFS_wValueLo) == 0xEE )
                    {
                        pStr = &USBFS_MSOS_DESCRIPTOR[0];
                    }
                #endif /* End USBFS_ENABLE_MSOS_STRING*/
                /* SN string*/
                #if defined(USBFS_ENABLE_SN_STRING)
                    if( CY_GET_REG8(USBFS_wValueLo) == 
                        USBFS_DEVICE0_DESCR[USBFS_DEVICE_DESCR_SN_SHIFT] )
                    {
                        pStr = &USBFS_SN_STRING_DESCRIPTOR[0];
                        if(USBFS_bSNStringConfirm != 0)
                        {
                            pStr = USBFS_bFWSerialNumberStringDescriptor;
                        }	
                        #if defined(USBFS_ENABLE_IDSN_STRING)
                        /* TODO: read DIE ID and genarete string descriptor in RAM*/
                            USBFS_ReadDieID(USBFS_bIDSerialNumberStringDescriptor);
                            pStr = USBFS_bIDSerialNumberStringDescriptor;
                        #endif	
                    }
                #endif	/* End USBFS_ENABLE_SN_STRING */ 
                    if (*pStr != 0)
                    {
                        CurrentTD.wCount    = *pStr;
                        CurrentTD.pData     = pStr;
                        bRequestHandled     = USBFS_InitControlRead();
                    }
                }
            #endif /* End USBFS_ENABLE_STRINGS*/
            else
            {
                bRequestHandled = USBFS_DispatchClassRqst();
            }
            break;
        case USBFS_GET_STATUS:
            switch ((CY_GET_REG8(USBFS_bmRequestType) & USBFS_RQST_RCPT_MASK))
            {
            case USBFS_RQST_RCPT_EP:
                    CurrentTD.wCount = 1;
                    CurrentTD.pData  = &USBFS_EP[CY_GET_REG8(USBFS_wIndexLo) & 0x07].bHWEPState;
                    bRequestHandled  = USBFS_InitControlRead();
                break;
            case USBFS_RQST_RCPT_DEV:
                CurrentTD.wCount    = 2;
                tBuffer[0] = USBFS_bDeviceStatus;
                tBuffer[1] = 0;
                CurrentTD.pData     = &tBuffer[0];
                bRequestHandled     = USBFS_InitControlRead();
                break;
            }
            break;
        case USBFS_GET_CONFIGURATION:
            CurrentTD.wCount    = 1;
            CurrentTD.pData     = &USBFS_bConfiguration;
            bRequestHandled     = USBFS_InitControlRead();
            break;
        default:
            break;
        }
    }
    else {
        /* Control Write */
        switch (CY_GET_REG8(USBFS_bRequest)) 
        {
        case USBFS_SET_ADDRESS:
            USBFS_bDeviceAddress = (CY_GET_REG8(USBFS_wValueLo));
            bRequestHandled = USBFS_InitNoDataControlTransfer();
            break;
        case USBFS_SET_CONFIGURATION:
            USBFS_bConfiguration	= CY_GET_REG8(USBFS_wValueLo);
            USBFS_Config();
            bRequestHandled	= USBFS_InitNoDataControlTransfer();
            break;
		case USBFS_SET_INTERFACE:
			if (USBFS_ValidateAlternateSetting())
			{
	            bRequestHandled	= USBFS_InitNoDataControlTransfer();
			}
			break;
		case USBFS_CLEAR_FEATURE:
            switch ((CY_GET_REG8(USBFS_bmRequestType) & USBFS_RQST_RCPT_MASK))
            {
            case USBFS_RQST_RCPT_EP:
                if (CY_GET_REG8(USBFS_wValueLo) ==USBFS_ENDPOINT_HALT)
                {
                    bRequestHandled	= USBFS_ClearEndpointHalt();
                }
                break;
            case USBFS_RQST_RCPT_DEV:
                /* Clear device REMOTE_WAKEUP */
                USBFS_bDeviceStatus &= ~USBFS_DEVICE_STATUS_REMOTE_WAKEUP;
                bRequestHandled	= USBFS_InitNoDataControlTransfer();
                break;
            }
            break;
		case USBFS_SET_FEATURE:
            switch ((CY_GET_REG8(USBFS_bmRequestType) & USBFS_RQST_RCPT_MASK))
            {
            case USBFS_RQST_RCPT_EP:
                if (CY_GET_REG8(USBFS_wValueLo) ==USBFS_ENDPOINT_HALT)
                {
                    bRequestHandled	= USBFS_SetEndpointHalt();
                }
                break;
            case USBFS_RQST_RCPT_DEV:
                /* Clear device REMOTE_WAKEUP */
                USBFS_bDeviceStatus &= ~USBFS_DEVICE_STATUS_REMOTE_WAKEUP;
                bRequestHandled	= USBFS_InitNoDataControlTransfer();
                break;
            }
        default:
            break;
        }
	}
	return bRequestHandled;
}    

#if defined(USBFS_ENABLE_IDSN_STRING)


    /***************************************************************************
    * Function Name: USBFS_ReadDieID
    ****************************************************************************
    * Summary:
    *   This routine read Die ID and genarete Serian Number string descriptor
    *   
    * Parameters:  
    *   uint8 *descr - pointer on string descriptor
    ***************************************************************************/
    void USBFS_ReadDieID(uint8 *descr)
    {
        uint8 i;
        /* check descriptor validation */
        if( (descr[0] > 1 ) &&   /* Descriptor Length */
            (descr[1] == 3) )    /* DescriptorType: STRING */    
        {
            /* fill descriptor by counter for test only*/
            for(i=2; i < descr[0]; i+=2)
            descr[i] = 0x30 + ((i/2) & 0x7);
        }
    }

#endif /* End $INSTANCE_NAME`_ENABLE_IDSN_STRING*/


/*******************************************************************************
* Function Name: USBFS_Config
********************************************************************************
* Summary:
*   This routine configures endpoints for the entire configuration by scanning
*   the configuration descriptor.  It configures the bAlternateSetting 0 for
*   each interface.
*   
* Parameters:  
*   None
*******************************************************************************/
void USBFS_Config()
{
    uint8 ep,cur_ep,i;
    uint8 iso;
    uint16 wCount;
    T_USBFS_LUT *pTmp;
    T_USBFS_EP_SETTINGS_BLOCK *pEP;

    /* Clear all of the endpoints */
    for (ep = 0; ep < 9; ep++)
    {
        USBFS_EP[ep].bAttrib = 0;
        USBFS_EP[ep].bHWEPState = 0;
        USBFS_EP[ep].bAPIEPState = USBFS_NO_EVENT_PENDING;
        USBFS_EP[ep].bEPToggle = 0;
        USBFS_EP[ep].bEPMode = 0;
        USBFS_EP[ep].wBufferSize = 0;
		USBFS_bInterfaceSetting[ep] = 0x00;
    }

    pTmp = USBFS_GetConfigTablePtr(USBFS_bConfiguration - 1);
    pTmp++;
    ep = pTmp->c;                                       /* For this table, c is the number of endpoints  */
    pEP = (T_USBFS_EP_SETTINGS_BLOCK *) pTmp->p_list;     /* and p_list points the endpoint setting table. */
    
    for (i = 0; i < ep; i++, pEP++)
    {
        cur_ep = pEP->bAddr; 
        cur_ep &= 0x0F;
        iso  = ((pEP->bmAttributes & USBFS_EP_TYPE_MASK) == USBFS_EP_TYPE_ISOC);
        if (pEP->bAddr & USBFS_DIR_IN)
        {
        /* IN Endpoint */
            USBFS_EP[cur_ep].bAPIEPState = USBFS_EVENT_PENDING;
            USBFS_EP[cur_ep].bEPMode = (iso ? USBFS_MODE_ISO_IN : USBFS_MODE_ACK_IN);
        }
        else
        {
        /* OUT Endpoint */
            USBFS_EP[cur_ep].bAPIEPState = USBFS_NO_EVENT_PENDING;
            USBFS_EP[cur_ep].bEPMode = (iso ? USBFS_MODE_ISO_OUT : USBFS_MODE_ACK_OUT);
        }
        USBFS_EP[cur_ep].wBufferSize = pEP->wBufferSize;
        USBFS_EP[cur_ep].bAddr = pEP->bAddr;
    } 

    /* Set the endpoint buffer addresses */
    wCount = 0;
    ep = 1;
    for (i = 0; i < 0x80; i+= 0x10)
    {
        CY_SET_REG8(&USBFS_ARB_EP1_CFG[i], USBFS_ARB_EPX_CFG_CRC_BYPASS);
        
        
        CY_SET_REG8(&USBFS_SIE_EP1_CR0[i], USBFS_MODE_NAK_IN_OUT);

        USBFS_EP[ep].wBuffOffset = wCount;
        wCount += USBFS_EP[ep].wBufferSize;
        CY_SET_REG8(&USBFS_SIE_EP1_CNT0[i],  (USBFS_EP[ep].wBufferSize >> 8));
        CY_SET_REG8(&USBFS_SIE_EP1_CNT1[i],  USBFS_EP[ep].wBufferSize & 0xFFu);

        CY_SET_REG8(&USBFS_ARB_RW1_RA[i],        USBFS_EP[ep].wBuffOffset & 0xFFu);
        CY_SET_REG8(&USBFS_ARB_RW1_RA_MSB[i],    (USBFS_EP[ep].wBuffOffset >> 8));
        CY_SET_REG8(&USBFS_ARB_RW1_WA[i],        USBFS_EP[ep].wBuffOffset & 0xFFu);
        CY_SET_REG8(&USBFS_ARB_RW1_WA_MSB[i],    (USBFS_EP[ep].wBuffOffset >> 8));
        
        ep++;
    }
    CY_SET_REG8(USBFS_SIE_EP_INT_EN, 0xFF);
}


/*******************************************************************************
* Function Name: USBFS_GetConfigTablePtr
********************************************************************************
* Summary:
*   This routine returns a pointer a configuration table entry
*   
* Parameters:  
*   c Configuration Index
*******************************************************************************/
T_USBFS_LUT *USBFS_GetConfigTablePtr(uint8 c)
{
    /* Device Table */
    T_USBFS_LUT *pTmp = USBFS_GetDeviceTablePtr();

    /* The first entry points to the Device Descriptor, the the configuration entries  */
    return pTmp[c + 1].p_list;
}


/*******************************************************************************
* Function Name: USBFS_GetDeviceTablePtr
********************************************************************************
* Summary:
*   This routine returns a pointer to the current config table based on the
*   selector in the request wValueLo
*   
* Parameters:  
*   None
*******************************************************************************/
T_USBFS_LUT *USBFS_GetDeviceTablePtr()
{
    /* Device Table */
    return USBFS_TABLE[USBFS_bDevice].p_list;
}


/*******************************************************************************
* Function Name: USBFS_SetEndpointHalt
********************************************************************************
* Summary:
*   This routine handles set endpoint halt
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_SetEndpointHalt()
{
    uint8 ep, ri;
    uint8 bRequestHandled = USBFS_FALSE;

    /* Clear endpoint halt */
    ep = (CY_GET_REG8(USBFS_wIndexLo) & 0x07);
    ri = ((ep - 1) << 4);
    if (ep != 0)
    {
        /* Set the endpoint Halt */
        USBFS_EP[ep].bHWEPState |= (USBFS_ENDPOINT_STATUS_HALT);
        
        /* Clear the data toggle */
        USBFS_EP[ep].bEPToggle = 0;
        
        if (USBFS_EP[ep].bAddr & USBFS_DIR_IN)
        {
            /* IN Endpoint */
            USBFS_EP[ep].bAPIEPState = USBFS_NO_EVENT_ALLOWED;
            CY_SET_REG8(&USBFS_SIE_EP1_CR0[ri], (USBFS_MODE_STALL_DATA_EP | 
                                                            USBFS_MODE_ACK_IN));
        }
        else
        {
            /* OUT Endpoint */
            USBFS_EP[ep].bAPIEPState = USBFS_NO_EVENT_ALLOWED;
            CY_SET_REG8(&USBFS_SIE_EP1_CR0[ri], (USBFS_MODE_STALL_DATA_EP | 
                                                            USBFS_MODE_ACK_OUT));
        }
        bRequestHandled = USBFS_InitNoDataControlTransfer();
    }
    return bRequestHandled;
}


/*******************************************************************************
* Function Name: USBFS_ClearEndpointHalt
********************************************************************************
* Summary:
*   This routine handles clear endpoint halt
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_ClearEndpointHalt()
{
    uint8 ep, ri;
    uint8 bRequestHandled = USBFS_FALSE;

    /* Clear endpoint halt */
    ep = (CY_GET_REG8(USBFS_wIndexLo) & 0x07);
    ri = ((ep  - 1) << 4);

    if (ep != 0)
    {
        /* Set the endpoint Halt */
        USBFS_EP[ep].bHWEPState &= ~(USBFS_ENDPOINT_STATUS_HALT);
        
        /* Clear the data toggle */
        USBFS_EP[ep].bEPToggle = 0;
        
        if (USBFS_EP[ep].bAddr & USBFS_DIR_IN)
        {
            /* IN Endpoint */
            USBFS_EP[ep].bAPIEPState = USBFS_NO_EVENT_PENDING;
            CY_SET_REG8(&USBFS_SIE_EP1_CR0[ri], USBFS_MODE_NAK_IN);
        }
        else
        {
            /* OUT Endpoint */
            USBFS_EP[ep].bAPIEPState = USBFS_EVENT_PENDING;
            CY_SET_REG8(&USBFS_SIE_EP1_CR0[ri], USBFS_MODE_ACK_OUT);
        }
        bRequestHandled = USBFS_InitNoDataControlTransfer();
    }
    return bRequestHandled;
}


/*******************************************************************************
* Function Name: USBFS_ValidateAlternateSetting
********************************************************************************
* Summary:
*   Validates (and records) a SET INTERFACE request
*   
* Parameters:  
*   None
*******************************************************************************/
uint8 USBFS_ValidateAlternateSetting(void)
{
    uint8 bRequestHandled = USBFS_TRUE;

	// TODO: Validate interface setting, stall if invalid.
	USBFS_bInterfaceSetting[CY_GET_REG8(USBFS_wIndexLo)] = CY_GET_REG8(USBFS_wValueLo);
	return bRequestHandled;
}
