/* --------------------------------------------------------------------------
* Copyright 2010, Cypress Semiconductor Corporation.
*
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international
* treaty provisions. Cypress hereby grants to licensee a personal,
* non-exclusive, non-transferable license to copy, use, modify, create
* derivative works of, and compile the Cypress Source Code and derivative
* works for the sole purpose of creating custom software in support of
* licensee product to be used only in conjunction with a Cypress integrated
* circuit as specified in the applicable agreement. Any reproduction,
* modification, translation, compilation, or representation of this
* software except as specified above is prohibited without the express
* written permission of Cypress.
* 
* Disclaimer: CYPRESS MAKES NO WARRANTY OF ANY KIND,EXPRESS OR IMPLIED,
* WITH REGARD TO THIS MATERIAL, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
* Cypress reserves the right to make changes without further notice to the
* materials described herein. Cypress does not assume any liability arising
* out of the application or use of any product or circuit described herein.
* Cypress does not authorize its products for use as critical components in
* life-support systems where a malfunction or failure may reasonably be
* expected to result in significant injury to the user. The inclusion of
* Cypress' product in a life-support systems application implies that the
* manufacturer assumes all risk of such use and in doing so indemnifies
* Cypress against all charges.
* 
* Use may be limited by and subject to the applicable Cypress software
* license agreement.
* -----------------------------------------------------------------------------
* Copyright (c) Cypress MicroSystems 2000-2003. All Rights Reserved.
*
*****************************************************************************
*  Project Name: Example_USB  
*  Device Tested: CY8C3866AXI, CY8C5588AXI
*  Software Version: PSoC Creator FCS 1.0
*  Complier tested: Keil(C51) and GCC
*  Related Hardware: CY8CKIT-001
*****************************************************************************
***************************************************************************** */

/* 
 * Description:
 * This project is aimed at transferring 64 or less bytes of data from the PC to the PSoC 3/5 device 
 * using the USB interrupt Out endpoint. The data sent is modified by the PSoC 3/5 device and is 
 * sent back to the PC using interrupt In endpoint. A GUI is created for sending data from the PC to the 
 * PSoC 3/5 device and to read back the data from the PSoC 3/5 device. 
 * A full speed USB component in PSoC 3/5 is used in this project which enumerates as a generic HID class device
 * The project needs the connection of the USB cable from the DVK to the PC.
 * The USB uses P15[6] and P15[7]. Once the component is placed on the TopDesign, these pins are automatically
 * blocked. The data from the USB transceiver is then fed to the USB cable which is connected to J9 in DVK
*/

#include <device.h>

#define TRUE 1				
#define FALSE 0

/* Macro to store the Device Id */
#define DEVICE_ID 0 

/* Macro to store the Out Endpoint number */
#define OUT_ENDPOINT 2    
/* Macro to store the In Endpoint number */
#define IN_ENDPOINT 1     

/* Macro to store the maximum number of bytes that can be received */
#define MAX_NUM_BYTES  64

/* Function to process data at the endpoints */
void ProcessEP2Data(void);

/* This variable is declarded in the USBFS_drv.c file.
 * This variable holds the configuration data for the endpoints */
extern T_USBFS_EP_CTL_BLOCK USBFS_EP[];

/* This variable is accesed inside the WSB Interrupt (see file USBDS_episr.c)
 * The variable is set to 1 when the interrupt occurs (The endpoint 2 ISR is named as USBFS_EP_2_ISR) 
 * and it is set to 0, once the data processing is completed in main.c */
uint8 USB_interruptFlag = 0;

void main()
{
	/* Enable the global interrupts */
	CYGlobalIntEnable; 
    
    /* Start USBFS Operation for the DEVICE_ID and with 3.3V operation  */
    USBFS_Start(DEVICE_ID, USBFS_3V_OPERATION); 
                                
	/* Wait for Device to enumerate. The function will return a 0 until the enumeration is complete */
    while(FALSE == USBFS_bGetConfiguration())
	{
		/* Waiting for enumeration */
		;
	}
	
	/* Once the device is enumerated, Enable Endpoint 2 for receiving data */
	USBFS_EnableOutEP(OUT_ENDPOINT);  
	
    while(TRUE)
    {
		/* Program Loop */
        /* Check if the Interrupt has occurred */
		if(USB_interruptFlag == TRUE)
		{
			/* Clear the Interrupt flag */
            USB_interruptFlag=0;
            
			ProcessEP2Data();
		}
    }
}

/* 
 * Summary: 
 *  This function copies the received data bytes from Out endpoint,
 *  Increments the data and puts it back to the In endpoint buffer
 * Parameters:
 *  None
 * Return
 *  None
 * Other details:
 *  This function is called when the Out endpoint ISR occurs */
 
void ProcessEP2Data()
{
    uint8 count, index;
	uint8 USB_data[MAX_NUM_BYTES];
	
    /* Get the number of data received */
    count = USBFS_wGetEPCount(OUT_ENDPOINT);
   
    /* Read the Out endpoint data into a user buffer (USB_data) */
    USBFS_ReadOutEP(OUT_ENDPOINT, USB_data, count);
    
    /* Enable Out Endpoint, to receive next set of data */
    USBFS_EnableOutEP(OUT_ENDPOINT);
    
    /* All the received data is incremented by 1 by PSoC before sending back to the PC */
    for(index = 0; index < count; index++)
    {
        /* Increment the received data */
        USB_data[index]++;
    }
    
    /* Load the incremented data buffer back to in endpoint.
    * This data will be received by the PC */
    USBFS_LoadEP(IN_ENDPOINT, USB_data, count);
   
}
/* [] END OF FILE */
